# X-App-SDK Integration Guide for Mini App

## Tổng quan

Tài liệu này mô tả cách tích hợp mini app với `x-app-sdk` để lấy thông tin token, user từ Super App. Super App đã có sẵn `x-app-sdk`, mini app chỉ cần gọi các method để lấy dữ liệu.

## Cài đặt

### 1. Build Flutter web app

```bash
flutter build web
```

## Cách sử dụng

### Trong Super App

Super App đã có sẵn `x-app-sdk` và cung cấp các method global:

```javascript
// Super App đã có sẵn các method này:
// window.getToken() - Lấy token đăng nhập
// window.getInfo(key) - Lấy thông tin người dùng theo key

// Mini app sẽ tự động gọi:
// window.getToken() - Lấy token
// window.getInfo('USER_ID') - Lấy ID người dùng
// window.getInfo('USER_NAME') - Lấy tên người dùng
// window.getInfo('USER_EMAIL') - Lấy email người dùng
// window.getInfo('USER_PHONE') - Lấy số điện thoại
```

### Trong Mini App (Flutter)

Mini app sẽ tự động lấy dữ liệu từ Super App:

```dart
// Lấy token
String? token = webGetAppHostToken();

// Lấy user info
Map<String, dynamic>? user = webGetAppHostUser();

// Kiểm tra data có sẵn
bool isReady = webIsAppHostDataReady();

// Kiểm tra x-app-sdk có sẵn từ Super App
bool sdkAvailable = webIsSDKAvailable();

// Lấy error message nếu có
String? error = webGetAppHostError();

// Lấy thông tin user theo key cụ thể
dynamic userInfo = await webGetUserInfoByKey('USER_NAME');

// Lấy token bất đồng bộ
String? token = await webGetTokenAsync();
```

## API Reference

### Web Helper Functions

- `webGetAppHostToken()`: Lấy token từ Super App
- `webGetAppHostUser()`: Lấy thông tin user từ Super App
- `webIsAppHostDataReady()`: Kiểm tra data có sẵn
- `webIsSDKAvailable()`: Kiểm tra x-app-sdk có sẵn từ Super App
- `webGetAppHostError()`: Lấy error message
- `webInitializeXAppSDK()`: Khởi tạo x-app-sdk service
- `webStoreAppHostData(token, user)`: Lưu data vào localStorage
- `webClearAppHostData()`: Xóa data
- `webCallXAppSDKMethod(methodName, args)`: Gọi method của x-app-sdk
- `webGetUserInfoByKey(key)`: Lấy thông tin user theo key
- `webGetTokenAsync()`: Lấy token bất đồng bộ

### XAppSDKService

```dart
final service = XAppSDKService();

// Khởi tạo
await service.initialize();

// Lấy dữ liệu
String? token = service.token;
Map<String, dynamic>? user = service.user;
bool isReady = service.isReady;
String? error = service.error;

// Lưu dữ liệu
service.storeData(token, user);

// Xóa dữ liệu
service.clearData();

// Gọi method SDK trực tiếp
dynamic result = await service.callSDKMethod('getInfo', ['USER_NAME']);

// Lấy thông tin user theo key
dynamic userInfo = await service.getUserInfo('USER_EMAIL');

// Lấy token bất đồng bộ
String? token = await service.getTokenAsync();

// Kiểm tra SDK có sẵn
bool sdkAvailable = service.isSDKAvailable();
```

## X-App-SDK Methods (từ Super App)

### getToken()
Lấy token đăng nhập người dùng từ Super App.

```javascript
// Super App cung cấp method này
window.getToken().then(token => {
    console.log('Token người dùng:', token);
}).catch(error => {
    console.error('Lỗi lấy token:', error);
});
```

### getInfo(key)
Lấy thông tin người dùng từ Super App theo key.

```javascript
// Super App cung cấp method này
// Lấy user ID
window.getInfo('USER_ID').then(info => {
    console.log('ID người dùng:', info.data);
}).catch(error => {
    console.error('Lỗi lấy thông tin:', error);
});

// Lấy thông tin khác
window.getInfo('USER_NAME').then(info => {
    console.log('Tên người dùng:', info.data);
});
```

### Available Keys
- `'USER_ID'`: ID người dùng
- `'USER_NAME'`: Tên người dùng  
- `'USER_EMAIL'`: Email người dùng
- `'USER_PHONE'`: Số điện thoại

## Fallback Methods

Nếu `x-app-sdk` không khả dụng, app sẽ thử các phương pháp fallback:

1. **URL Parameters**: `?token=xxx&user=yyy`
2. **localStorage**: `app_host_token`, `app_host_user`

## Debugging

Để debug, mở Developer Tools và kiểm tra:

1. Console logs với prefix `🔍`, `✅`, `❌`, `⚠️`
2. `window.AppHostData` object
3. Network tab để xem việc load x-app-sdk

## Lưu ý

- Chỉ hoạt động trên web platform
- Super App đã có sẵn `x-app-sdk` library
- Mini app gọi `window.getToken()` và `window.getInfo()` từ Super App
- Data được lưu trong localStorage để sử dụng lại
- App tự động listen cho updates từ Super App
- SDK với dữ liệu thật chỉ được trả ra khi chạy trên Super App
- Khi chạy trên web thường sẽ có data mẫu có cấu trúc tương tự data thật

## Troubleshooting

### Lỗi thường gặp:

1. **"x-app-sdk not available from Super App"**: Super App chưa expose methods
2. **"No data available from Super App"**: Không lấy được data từ Super App
3. **CORS errors**: Cần cấu hình CORS cho Super App

### Giải pháp:

1. Kiểm tra Super App đã expose `window.getToken()` và `window.getInfo()` chưa
2. Kiểm tra console logs để xem chi tiết lỗi
3. Sử dụng fallback methods nếu cần
4. Kiểm tra `webIsSDKAvailable()` để xác nhận SDK có sẵn
