// Web-specific implementations
// ignore: avoid_web_libraries_in_flutter
import 'dart:convert';
import 'package:universal_html/html.dart' as html;
import 'x_app_sdk_service.dart';

void webReplaceUrl(String path) {
  try {
    final origin = html.window.location.origin;
    final newUrl = '$origin$path';
    html.window.history.replaceState(null, html.document.title ?? '', newUrl);
  } catch (_) {}
}

void webClearStorage() {
  try {
    html.window.localStorage.clear();
    html.window.sessionStorage.clear();
    // Optionally clear caches if any
  } catch (_) {}
}

/// Get token from app host via x-app-sdk
String? webGetAppHostToken() {
  try {
    return XAppSDKService().getToken();
  } catch (e) {
    print('❌ Error getting app host token: $e');
    return null;
  }
}

/// Get user info from app host via x-app-sdk
Map<String, dynamic>? webGetAppHostUser() {
  try {
    return XAppSDKService().getUser();
  } catch (e) {
    print('❌ Error getting app host user: $e');
    return null;
  }
}

/// Check if app host data is ready
bool webIsAppHostDataReady() {
  try {
    return XAppSDKService().isServiceReady;
  } catch (e) {
    print('❌ Error checking app host data ready: $e');
    return false;
  }
}

/// Get error message from app host
String? webGetAppHostError() {
  try {
    return XAppSDKService().getErrorMessage();
  } catch (e) {
    print('❌ Error getting app host error: $e');
    return null;
  }
}

/// Initialize x-app-sdk service
Future<void> webInitializeXAppSDK() async {
  try {
    await XAppSDKService().initialize();
    XAppSDKService().listenForUpdates();
  } catch (e) {
    print('❌ Error initializing x-app-sdk: $e');
  }
}

/// Store app host data
void webStoreAppHostData(String token, Map<String, dynamic>? user) {
  try {
    XAppSDKService().storeData(token, user);
  } catch (e) {
    print('❌ Error storing app host data: $e');
  }
}

/// Clear app host data
void webClearAppHostData() {
  try {
    XAppSDKService().clearData();
  } catch (e) {
    print('❌ Error clearing app host data: $e');
  }
}

/// Execute JavaScript in the web context
Future<dynamic> webExecuteJavaScript(String script) async {
  try {
    // For now, we'll use a simpler approach
    // This method is mainly for future extensibility
    print('⚠️ webExecuteJavaScript is not fully implemented yet');
    return null;
  } catch (e) {
    print('❌ Error executing JavaScript: $e');
    return null;
  }
}

/// Call x-app-sdk method if available
Future<dynamic> webCallXAppSDKMethod(String methodName, [List<dynamic>? args]) async {
  try {
    return await XAppSDKService().callSDKMethod(methodName, args);
  } catch (e) {
    print('❌ Error calling x-app-sdk method $methodName: $e');
    return null;
  }
}

/// Get user info by key from app host
Future<dynamic> webGetUserInfoByKey(String key) async {
  try {
    return await XAppSDKService().getUserInfo(key);
  } catch (e) {
    print('❌ Error getting user info by key $key: $e');
    return null;
  }
}

/// Get token asynchronously from app host
Future<String?> webGetTokenAsync() async {
  try {
    return await XAppSDKService().getTokenAsync();
  } catch (e) {
    print('❌ Error getting token async: $e');
    return null;
  }
}

/// Check if x-app-sdk is available from Super App
bool webIsSDKAvailable() {
  try {
    return XAppSDKService().isSDKAvailable();
  } catch (e) {
    print('❌ Error checking SDK availability: $e');
    return false;
  }
}

/// Close app and return to Super App
void webCloseApp([Map<String, dynamic>? data]) {
  try {
    XAppSDKService().closeApp(data);
  } catch (e) {
    print('❌ Error closing app: $e');
  }
}


