/// Centralized validation utilities
class ValidationUtils {
  /// Validate phone number with Vietnamese format
  static bool isValidPhoneNumber(String phone) {
    final cleanPhone = phone.replaceAll(RegExp(r'\s+'), '');
    final regex = RegExp(r'^(0|\+84)(3[2-9]|5[6|8|9]|7[0|6-9]|8[1-5]|9[0-4|6-9])[0-9]{7}$');
    return regex.hasMatch(cleanPhone);
  }
  
  /// Validate email format
  static bool isValidEmail(String email) {
    final emailRegex = RegExp(r'^[\w-\.]+@([\w-]+\.)+[\w-]{2,4}$');
    return emailRegex.hasMatch(email);
  }
  
  /// Validate password strength
  static bool isValidPassword(String password) {
    // At least 6 characters
    if (password.length < 6) return false;
    
    // Contains at least one letter and one number
    final hasLetter = RegExp(r'[a-zA-Z]').hasMatch(password);
    final hasNumber = RegExp(r'[0-9]').hasMatch(password);
    
    return hasLetter && hasNumber;
  }
  
  /// Get phone number error message
  static String? getPhoneNumberError(String phone) {
    if (phone.isEmpty) return 'Vui lòng nhập số điện thoại';
    if (!isValidPhoneNumber(phone)) return 'Số điện thoại không hợp lệ';
    return null;
  }
  
  /// Get email error message
  static String? getEmailError(String email) {
    if (email.isEmpty) return 'Vui lòng nhập email';
    if (!isValidEmail(email)) return 'Email không hợp lệ';
    return null;
  }
  
  /// Get password error message
  static String? getPasswordError(String password) {
    if (password.isEmpty) return 'Vui lòng nhập mật khẩu';
    if (!isValidPassword(password)) return 'Mật khẩu phải có ít nhất 6 ký tự và bao gồm cả chữ và số';
    return null;
  }
}

