const SDK_CANDIDATE_PATHS = [
  './x_app_sdk_bundle.js',
  '../js/x_app_sdk_bundle.js',
  '../node_modules/x-app-sdk/dist/index.es.js',
];

function ensureSdkShape(module, source) {
  if (!module) {
    throw new Error(`Module from ${source} is undefined`);
  }
  if (typeof module.getToken !== 'function' || typeof module.closeApp !== 'function') {
    throw new Error(`Module from ${source} is missing required exports (getToken, closeApp)`);
  }
}

export async function bootstrapXAppSdk() {
  let lastError;

  for (const path of SDK_CANDIDATE_PATHS) {
    try {
      const module = await import(path);
      ensureSdkShape(module, path);

      window.xAppSdk = {
        getToken: module.getToken,
        closeApp: module.closeApp,
      };

      document.dispatchEvent(
        new CustomEvent('xAppSdkReady', {
          detail: { source: path },
        }),
      );

      console.log(`✅ x-app-sdk loaded from ${path}`);
      console.log('🔧 SDK methods available:', Object.keys(window.xAppSdk));

      return window.xAppSdk;
    } catch (error) {
      lastError = error;
      console.warn(`ℹ️ Failed to load x-app-sdk from ${path}`, error);
    }
  }

  const message = 'Unable to locate x-app-sdk module';
  console.error(`❌ ${message}`);
  throw lastError ?? new Error(message);
}

export async function loadXAppSdkModule() {
  const sdk = await bootstrapXAppSdk();
  return sdk;
}
