import 'package:flutter/material.dart';
import 'package:flutter/foundation.dart';
import 'package:get/get.dart';
import 'package:mypoint_flutter_app/shared/widgets/image_loader.dart';
import '../../shared/widgets/base_view/base_screen.dart';
import '../../shared/widgets/base_view/basic_state.dart';
import '../../app/routing/directional_action_type.dart';
import '../../shared/router_gage.dart';
import '../../shared/widgets/back_button.dart';
import '../../shared/widgets/custom_empty_widget.dart';
import '../../shared/widgets/custom_navigation_bar.dart';
import '../popup_manager/popup_runner_helper.dart';
import 'game_tab_viewmodel.dart';

class GameTabScreen extends BaseScreen {
  const GameTabScreen({super.key});

  @override
  State<GameTabScreen> createState() => _GameTabScreenState();
}

class _GameTabScreenState extends BaseState<GameTabScreen> with BasicState, PopupOnInit {
  final GameTabViewModel _viewModel = Get.put(GameTabViewModel());
  final LayerLink _layerLink = LayerLink();
  final GlobalKey _infoKey = GlobalKey();
  OverlayEntry? _popupEntry;
  bool _isPopupShown = false;
  late var _canBackButton = false;

  @override
  void initState() {
    super.initState();
    final args = Get.arguments;
    if (args is Map) {
      _canBackButton = args['can_back_button'] as bool? ?? false;
    }
    _viewModel.getGames();
    _viewModel.onShowAlertError = (message) {
      if (message.isNotEmpty) {
        showAlertError(content: message);
      }
    };
    _viewModel.gotoGameDetail = (data) {
      if (data.popup != null) {
        showPopup(data: data.popup!);
      } else {
        Get.toNamed(gameCardScreen, arguments: {'gameId': data.id});//'data': data,
      }
    };
    runPopupCheck(DirectionalScreenName.productVoucher);
  }

  @override
  Widget createBody() {
    return Scaffold(
      appBar: CustomNavigationBar(
        title: "Games",
        leftButtons: _canBackButton ? [CustomBackButton()] : [],
        rightButtons: [
          if (!kIsWeb && defaultTargetPlatform == TargetPlatform.iOS)
            CompositedTransformTarget(
              link: _layerLink,
              child: IconButton(
                key: _infoKey,
                icon: const Icon(Icons.info, color: Colors.white),
                onPressed: _togglePopup,
              ),
            ),
        ],
      ),
      body: Obx(() {
        final games = _viewModel.games;
        return RefreshIndicator(
          onRefresh: _viewModel.getGames,
          child: ListView.separated(
            physics: const AlwaysScrollableScrollPhysics(),
            padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 16),
            itemCount: games.isEmpty ? 1 : games.length + 1,
            separatorBuilder: (_, index) {
              if (games.isEmpty) return const SizedBox.shrink();
              if (index == 0) {
                return SizedBox(height: _viewModel.turnsNumberText.value.isEmpty ? 4 : 12);
              }
              return const SizedBox(height: 4);
            },
            itemBuilder: (context, index) {
              if (games.isEmpty) {
                return SizedBox(
                  height: MediaQuery.of(context).size.height * 0.8,
                  child: Center(child: EmptyWidget(isLoading: _viewModel.isLoading.value)),
                );
              }
              if (index == 0) {
                final turnsText = _viewModel.turnsNumberText.value;
                if (turnsText.isEmpty) {
                  return const SizedBox.shrink();
                }
                return Text(
                  turnsText,
                  style: const TextStyle(fontSize: 18, fontWeight: FontWeight.bold),
                );
              }
              final item = games[index - 1];
              return GestureDetector(
                onTap: () {
                  _viewModel.getGameDetail(item.id ?? "");
                },
                child: AspectRatio(
                  aspectRatio: 343 / 132,
                  child: ClipRRect(
                    borderRadius: BorderRadius.circular(16),
                    child: loadNetworkImage(
                      url: item.icon ?? '',
                      fit: BoxFit.fitWidth,
                      placeholderAsset: 'assets/images/bg_default_169.png',
                    ),
                  ),
                ),
              );
            },
          ),
        );
      }),
    );
  }

  void _togglePopup() {
    if (_isPopupShown) {
      _hidePopup();
    } else {
      _showPopup();
    }
  }

  void _showPopup() {
    final overlay = Overlay.of(context);
    final renderBox = _infoKey.currentContext?.findRenderObject() as RenderBox?;
    final offset = renderBox?.localToGlobal(Offset.zero) ?? Offset.zero;
    final size = renderBox?.size ?? Size.zero;
    final widthSize = MediaQuery.of(context).size.width * 0.85;

    _popupEntry = OverlayEntry(
      builder:
          (context) => Stack(
            children: [
              // 👉 Tap ngoài popup để close
              Positioned.fill(
                child: GestureDetector(
                  onTap: _hidePopup,
                  behavior: HitTestBehavior.translucent,
                  child: Container(color: Colors.transparent),
                ),
              ),
              Positioned(
                top: offset.dy + size.height + 8,
                left: widthSize * 0.15 / 0.85 / 2, // offset.dx - widthSize,
                child: Material(
                  borderRadius: BorderRadius.circular(16),
                  elevation: 4,
                  child: Container(
                    width: widthSize,
                    padding: const EdgeInsets.all(16),
                    decoration: BoxDecoration(color: Colors.white, borderRadius: BorderRadius.circular(16)),
                    child: const Text(
                      '1/ Thể lể trò chơi rất đơn giản, khách hàng sử dụng dịch vụ gói cước viễn thông của chúng tôi sẽ có cơ hội tham gia chơi game, khách hàng nào chơi cũng có thưởng do MyPoint (PayTech) tài trợ 100% phần quà.'
                      '\n\n2/ Ngoài ra các giải thưởng và luật lệ trong trò chơi không phải do Apple quản lý và tài trợ, điều này đã được thể hiện trong thể lệ và văn bản công bố trò chơi với khách hàng.',
                      style: TextStyle(fontSize: 14),
                    ),
                  ),
                ),
              ),
            ],
          ),
    );

    overlay.insert(_popupEntry!);
    _isPopupShown = true;
  }

  void _hidePopup() {
    _popupEntry?.remove();
    _popupEntry = null;
    _isPopupShown = false;
  }
}
