import 'dart:convert';
import 'dart:io';
import 'dart:typed_data';

import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:image_picker/image_picker.dart';
import 'package:flutter_image_compress/flutter_image_compress.dart';
import 'package:mypoint_flutter_app/shared/preferences/data_preference.dart';
import 'package:path_provider/path_provider.dart';
import 'package:permission_handler/permission_handler.dart';
import 'package:mypoint_flutter_app/features/personal/personal_edit_item_model.dart';
import 'package:mypoint_flutter_app/features/personal/personal_edit_viewmodel.dart';
import 'package:mypoint_flutter_app/features/personal/personal_gender.dart';
import 'package:mypoint_flutter_app/features/personal/widgets/avatar_picker_sheet.dart';
import 'package:mypoint_flutter_app/shared/widgets/image_loader.dart';
import 'package:mypoint_flutter_app/core/services/web/web_helper.dart';
import '../../core/services/web/web_info_data.dart';
import '../../shared/widgets/base_view/base_screen.dart';
import '../../shared/widgets/base_view/basic_state.dart';
import '../../core/theme/base_color.dart';
import '../../shared/widgets/alert/data_alert_model.dart';
import '../../shared/widgets/custom_navigation_bar.dart';

class PersonalEditScreen extends BaseScreen {
  const PersonalEditScreen({super.key});
  @override
  State<PersonalEditScreen> createState() => _PersonalEditScreenState();
}

class _PersonalEditScreenState extends BaseState<PersonalEditScreen> with BasicState {
  final viewModel = Get.put(PersonalEditViewModel());

  @override
  initState() {
    super.initState();
    viewModel.onShowAlertError = (message) {
      showAlertError(content: message, barrierDismissible: true, onConfirmed: null);
    };
    viewModel.updateProfileResponseSuccess = () {
      showAlertError(
          content: "Cập nhật thông tin cá nhân thành công!",
          headerImage: "assets/images/ic_pipi_05.png",
          onConfirmed: () => Get.back());
    };
  }

  @override
  Widget createBody() {
    return GestureDetector(
      onTap: () => FocusScope.of(context).unfocus(),
      behavior: HitTestBehavior.translucent,
      child: Scaffold(
        appBar: CustomNavigationBar(title: "Chỉnh sửa thông tin cá nhân"),
        body: Obx(() {
          List<PersonalEditItemModel> items;
          final editDataModel = viewModel.editDataModel.value;
          if (editDataModel == null) {
            return const SizedBox.shrink();
          }
          items = PersonalEditItemModel.personalEditFields(editDataModel);
          return CustomScrollView(
            physics: const AlwaysScrollableScrollPhysics(),
            slivers: [
              SliverToBoxAdapter(
                child: Padding(
                  padding: const EdgeInsets.symmetric(horizontal: 32, vertical: 12),
                  child: _buildAvatarItem(),
                ),
              ),
              SliverList(
                delegate: SliverChildBuilderDelegate((context, index) {
                  final item = items[index];
                  return _editPersonalItem(item);
                }, childCount: items.length),
              ),
            ],
          );
        }),
        bottomNavigationBar: _buildContinueButton(),
      ),
    );
  }

  Widget _buildContinueButton() {
    return Container(
      padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 16),
      decoration: const BoxDecoration(
        color: Colors.white,
        boxShadow: [BoxShadow(color: Colors.black54, blurRadius: 8, offset: Offset(0, 4))],
      ),
      child: SafeArea(
        top: false,
        child: Obx(() {
          return SizedBox(
            width: double.infinity,
            height: 48,
            child: ElevatedButton(
              onPressed:
                  viewModel.isValidate.value
                      ? () {
                        viewModel.updateProfile();
                      }
                      : null,
              style: ElevatedButton.styleFrom(
                backgroundColor: viewModel.isValidate.value ? BaseColor.primary500 : Colors.grey,
                shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(12)),
              ),
              child: const Text(
                'Cập Nhật Ngay',
                style: TextStyle(fontSize: 16, color: Colors.white, fontWeight: FontWeight.bold),
              ),
            ),
          );
        }),
      ),
    );
  }

  Widget _buildAvatarItem() {
    final avatar = viewModel.editDataModel.value?.avatar;
    final fallbackAvatar = WebData.getAvatar();
    return Center(
      child: GestureDetector(
        onTap: () {
          _showAvatarPicker();
        },
        child: Stack(
          alignment: Alignment.bottomRight,
          children: [
            ClipOval(
                child: _buildAvatarImage(
                    avatar?.isNotEmpty == true ? avatar : fallbackAvatar,
                    width: 100,
                    height: 100,
                    placeholderAsset: "assets/images/bg_default_11.png"
                )
            ),
            Positioned(
              bottom: 4,
              right: 4,
              child: Container(
                padding: const EdgeInsets.all(4),
                decoration: const BoxDecoration(color: Colors.red, shape: BoxShape.circle),
                child: const Icon(Icons.camera_alt, color: Colors.white, size: 18),
              ),
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildAvatarImage(String? avatar, {
    double? width,
    double? height,
    String placeholderAsset = "assets/images/bg_default_11.png",
  }) {
    if (avatar == null || avatar.isEmpty) {
      return Image.asset(
        placeholderAsset,
        fit: BoxFit.cover,
        width: width,
        height: height,
      );
    }
    if (avatar.contains("avatar-")) {
      return Image.asset(
        "assets/images/$avatar",
        fit: BoxFit.cover,
        width: width,
        height: height,
      );
    }
    if (avatar.startsWith("data:image")) {
      final bytes = _decodeBase64Image(avatar);
      if (bytes != null) {
        return Image.memory(
          bytes,
          fit: BoxFit.cover,
          width: width,
          height: height,
        );
      }
      return Image.asset(
        placeholderAsset,
        fit: BoxFit.cover,
        width: width,
        height: height,
      );
    }
    return loadNetworkImage(
      url: avatar,
      width: width,
      height: height,
      fit: BoxFit.cover,
      placeholderAsset: placeholderAsset,
    );
  }

  Uint8List? _decodeBase64Image(String value) {
    try {
      final data = value.split(',').last;
      return base64Decode(data);
    } catch (_) {
      return null;
    }
  }

  void _showAvatarPicker() {
    print("_showAvatarPicker ${viewModel.editDataModel.value?.avatar}");
    AvatarPickerSheet.show(
      context: context,
      onCameraTap: () {
        _pickImage("camera");
      },
      onGalleryTap: () {
        _pickImage("gallery");
      },
      onAvatarSelected: (avatarPath) {
        viewModel.editDataModel.value?.avatar = avatarPath;
        viewModel.editDataModel.refresh();
      },
      selectedAvatar: viewModel.editDataModel.value?.avatar,
    );
  }

  Future<void> _pickImage(String type) async {
    print("_pickImage _pickImage $type");
    if (kIsWeb) {
      final result = await webOpenPickerImage(type);
      final imageValue = _extractImageValue(result);
      final normalized = _normalizeImageValue(imageValue);
      if (normalized == null || normalized.isEmpty) {
        return;
      }
      viewModel.editDataModel.value?.avatar = normalized;
      viewModel.editDataModel.refresh();
      return;
    }

    final granted = await _requestNativePermission(type);
    if (!granted) {
      debugPrint("🚫 Permission denied for $type");
      return;
    }

    final picker = ImagePicker();
    final source = type == "camera" ? ImageSource.camera : ImageSource.gallery;
    final picked = await picker.pickImage(source: source);
    if (picked == null) {
      return;
    }
    final uploadPath = await _ensureUploadPng(picked.path);
    print("_pickImage type: $type, path: $uploadPath");
    await viewModel.uploadAvatarAndSet(uploadPath);
  }

  Future<String> _ensureUploadPng(String path) async {
    try {
      final bytes = await File(path).readAsBytes();
      final png = await FlutterImageCompress.compressWithList(
        bytes,
        format: CompressFormat.png,
        quality: 30,
        minWidth: 200,
        minHeight: 200,
      );
      if (png.isEmpty) {
        return path;
      }
      final dir = await getTemporaryDirectory();
      final fileName = "avatar_${DateTime.now().millisecondsSinceEpoch}.png";
      final outFile = File("${dir.path}/$fileName");
      await outFile.writeAsBytes(png, flush: true);
      return outFile.path;
    } catch (_) {
      return path;
    }
  }

  Future<bool> _requestNativePermission(String type) async {
    if (type == "camera") {
      final status = await Permission.camera.request();
      return status.isGranted;
    }
    final photosStatus = await Permission.photos.request();
    if (photosStatus.isGranted) {
      return true;
    }
    final storageStatus = await Permission.storage.request();
    return storageStatus.isGranted;
  }

  String? _extractImageValue(dynamic result) {
    if (result is String) {
      return result;
    }
    if (result is Map) {
      final data = result["data"] ?? result["image"] ?? result;
      if (data is String) {
        return data;
      }
      if (data is Map) {
        final base64 = data["base64"];
        if (base64 is String && base64.isNotEmpty) {
          return base64;
        }
        final path = data["path"];
        if (path is String && path.isNotEmpty) {
          return path;
        }
      }
    }
    return null;
  }

  String? _normalizeImageValue(String? value) {
    if (value == null || value.isEmpty) return null;
    if (value.startsWith("data:image") || value.startsWith("http") || value.startsWith("assets/")) {
      return value;
    }
    if (value.startsWith("/") || value.startsWith("file:")) {
      return value;
    }
    return "data:image/jpeg;base64,$value";
  }

  Future<void> _onTapItemChangeValue(PersonalEditItemModel item) async {
    if (item.sectionType == SectionPersonalEditType.province || item.sectionType == SectionPersonalEditType.district) {
      viewModel.navigateToLocationScreen(item);
    } else if (item.sectionType == SectionPersonalEditType.birthday) {
      if ((DataPreference.instance.profile?.workerSite?.birthday ?? "").isNotEmpty) return;
      final now = DateTime.now();
      final picked = await showDatePicker(
        context: context,
        initialDate: viewModel.birthday ?? now,
        firstDate: DateTime(1900),
        lastDate: DateTime(2100),
      );
      if (picked != null) {
        setState(() {
          viewModel.birthday = picked;
          viewModel.editDataModel.value?.birthday = picked;
          viewModel.isValidate.value = viewModel.validate();
        });
      }
    } else if (item.sectionType == SectionPersonalEditType.gender) {
      showGenderPicker(
        context: context,
        selected: viewModel.gender ?? PersonalGender.unknown,
        onSelected: (gender) {
          setState(() {
            viewModel.gender = gender;
            viewModel.editDataModel.value?.gender = gender;
          });
        },
      );
    }
  }

  Widget _editPersonalItem(PersonalEditItemModel item) {
    final isTapField =
        item.sectionType == SectionPersonalEditType.province ||
        item.sectionType == SectionPersonalEditType.district ||
        item.sectionType == SectionPersonalEditType.gender;
    final isDate = item.sectionType == SectionPersonalEditType.birthday;
    final isTappableItem = isTapField || isDate;
    final value = item.value ?? "";
    return Padding(
      padding: const EdgeInsets.only(top: 8, bottom: 8, left: 16, right: 16), // all(16.0),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          // const SizedBox(height: 8),
          RichText(
            text: TextSpan(
              text: item.title,
              style: const TextStyle(fontSize: 14, color: Colors.black, fontWeight: FontWeight.w600),
              children: [
                if (item.isRequired == true)
                  const TextSpan(text: ' (*)', style: TextStyle(color: Colors.red, fontWeight: FontWeight.bold)),
              ],
            ),
          ),
          const SizedBox(height: 6),
          GestureDetector(
            onTap: isTappableItem ? () => _onTapItemChangeValue(item) : null,
            child: Container(
              padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 12),
              decoration: BoxDecoration(
                color: item.isEditable == true ? Colors.white : Colors.grey.shade100,
                border: Border.all(color: Colors.grey.shade300),
                borderRadius: BorderRadius.circular(8),
                // color: Colors.grey.shade50,
              ),
              child: Row(
                children: [
                  Expanded(
                    child: TextField(
                      keyboardType: item.keyboardType ?? TextInputType.text,
                      controller: TextEditingController(text: value),
                      enabled: isTappableItem ? false : (item.isEditable ?? true),
                      decoration: InputDecoration.collapsed(
                        hintText: item.hintText ?? "",
                        hintStyle: TextStyle(color: Colors.grey, fontSize: 14),
                      ),
                      style: TextStyle(color: item.isEditable ?? true ? Colors.black87 : Colors.grey, fontSize: 16),
                      onChanged: (value) {
                        viewModel.updateItemEditData(item, value);
                        viewModel.isValidate.value = viewModel.validate();
                      },
                    ),
                  ),
                  if (item.showDropIcon == true) const Icon(Icons.expand_more, size: 20, color: Colors.grey),
                ],
              ),
            ),
          ),
          const SizedBox(height: 6),
          if (item.sectionType == SectionPersonalEditType.email && value.isNotEmpty && viewModel.isValidEmail(value) == false)
            Row(
              children: [
                const Icon(Icons.error, color: Colors.red, size: 14),
                const SizedBox(width: 4),
                Text('Email không đúng định dạng', style: const TextStyle(fontSize: 12, color: Colors.red)),
              ],
            ),
          if (item.warningText != null)
            Row(
              children: [
                const Icon(Icons.warning_amber_rounded, color: Colors.orange, size: 14),
                const SizedBox(width: 4),
                Text(item.warningText ?? '', style: const TextStyle(fontSize: 12, color: Colors.orange)),
              ],
            ),
          // const SizedBox(height: 12),
        ],
      ),
    );
  }

  void showGenderPicker({
    required BuildContext context,
    required PersonalGender selected,
    required Function(PersonalGender gender) onSelected,
  }) {
    showModalBottomSheet(
      context: context,
      backgroundColor: Colors.white,
      shape: const RoundedRectangleBorder(borderRadius: BorderRadius.vertical(top: Radius.circular(16))),
      builder: (_) {
        final genderList = PersonalGender.values.toList();
        return Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            Padding(
              padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
              child: Row(
                mainAxisAlignment: MainAxisAlignment.end,
                children: [
                  Expanded(
                    child: Center(
                      child: const Text('Giới tính', style: TextStyle(fontWeight: FontWeight.bold, fontSize: 16)),
                    ),
                  ),
                  GestureDetector(onTap: () => Navigator.of(context).pop(), child: const Icon(Icons.close, size: 20)),
                ],
              ),
            ),
            const Divider(height: 1),
            ...genderList.map((gender) {
              final isSelected = selected == gender;
              return ListTile(
                title: Text(
                  gender.display,
                  style: TextStyle(
                    fontWeight: isSelected ? FontWeight.bold : FontWeight.normal,
                    color: isSelected ? BaseColor.primary500 : Colors.black,
                  ),
                ),
                trailing: isSelected ? const Icon(Icons.check, color: BaseColor.primary500) : null,
                onTap: () {
                  Navigator.of(context).pop();
                  onSelected(gender);
                },
              );
            }),
            const Divider(height: 100),
          ],
        );
      },
    );
  }
}
