#!/usr/bin/env bash

# Script để tự động update imports sau khi migration
# Sử dụng find/replace để update các imports cũ sang mới

set -euo pipefail

PROJECT_ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
LIB_DIR="${PROJECT_ROOT}/lib"

# Colors
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

update_imports_phase1() {
    log_info "Updating imports for Phase 1..."
    
    # Find all Dart files
    find "${LIB_DIR}" -name "*.dart" -type f | while read -r file; do
        # Skip generated files
        if [[ "$file" == *.g.dart ]]; then
            continue
        fi
        
        # Update configs imports
        sed -i '' "s|package:mypoint_flutter_app/configs/|package:mypoint_flutter_app/core/config/|g" "$file"
        sed -i '' "s|package:mypoint_flutter_app/env_loader|package:mypoint_flutter_app/core/config/env_loader|g" "$file"
        
        # Update utils imports
        sed -i '' "s|package:mypoint_flutter_app/utils/|package:mypoint_flutter_app/core/utils/|g" "$file"
        sed -i '' "s|package:mypoint_flutter_app/shared/|package:mypoint_flutter_app/core/utils/|g" "$file"
        
        # Update extensions imports
        sed -i '' "s|package:mypoint_flutter_app/extensions/|package:mypoint_flutter_app/core/utils/extensions/|g" "$file"
        
        # Update navigation imports
        sed -i '' "s|package:mypoint_flutter_app/deferred/|package:mypoint_flutter_app/core/navigation/|g" "$file"
        sed -i '' "s|package:mypoint_flutter_app/directional/|package:mypoint_flutter_app/core/navigation/|g" "$file"
        sed -i '' "s|package:mypoint_flutter_app/base/app_navigator|package:mypoint_flutter_app/core/navigation/app_navigator|g" "$file"
        
        # Update resources/theme imports
        sed -i '' "s|package:mypoint_flutter_app/resources/base_color|package:mypoint_flutter_app/core/theme/colors|g" "$file"
        sed -i '' "s|package:mypoint_flutter_app/resources/text_style|package:mypoint_flutter_app/core/theme/text_styles|g" "$file"
        sed -i '' "s|package:mypoint_flutter_app/resources/button_style|package:mypoint_flutter_app/core/theme/button_styles|g" "$file"
        sed -i '' "s|package:mypoint_flutter_app/resources/define_image|package:mypoint_flutter_app/core/theme/images|g" "$file"
        sed -i '' "s|package:mypoint_flutter_app/resources/|package:mypoint_flutter_app/core/theme/|g" "$file"
        
        # Update core initialization imports
        sed -i '' "s|package:mypoint_flutter_app/core/app_initializer|package:mypoint_flutter_app/core/initialization/app_initializer|g" "$file"
        sed -i '' "s|package:mypoint_flutter_app/core/web_app_initializer|package:mypoint_flutter_app/core/initialization/web_app_initializer|g" "$file"
        sed -i '' "s|package:mypoint_flutter_app/core/deep_link_service|package:mypoint_flutter_app/core/initialization/deep_link_service|g" "$file"
    done
    
    log_info "Phase 1 imports updated!"
}

update_imports_phase2() {
    log_info "Updating imports for Phase 2..."
    
    find "${LIB_DIR}" -name "*.dart" -type f | while read -r file; do
        if [[ "$file" == *.g.dart ]]; then
            continue
        fi
        
        # Update networking imports
        sed -i '' "s|package:mypoint_flutter_app/networking/|package:mypoint_flutter_app/core/network/|g" "$file"
        
        # Update services imports
        sed -i '' "s|package:mypoint_flutter_app/services/|package:mypoint_flutter_app/core/services/auth/|g" "$file"
        
        # Update preference imports
        sed -i '' "s|package:mypoint_flutter_app/preference/|package:mypoint_flutter_app/core/storage/|g" "$file"
        
        # Update platform imports
        sed -i '' "s|package:mypoint_flutter_app/web/|package:mypoint_flutter_app/core/platform/web/|g" "$file"
        sed -i '' "s|package:mypoint_flutter_app/permission/|package:mypoint_flutter_app/core/platform/permission/|g" "$file"
        sed -i '' "s|package:mypoint_flutter_app/firebase/|package:mypoint_flutter_app/core/platform/firebase/|g" "$file"
    done
    
    log_info "Phase 2 imports updated!"
}

main() {
    local phase="${1:-}"
    
    if [ -z "$phase" ]; then
        echo "Usage: $0 [phase1|phase2]"
        echo ""
        echo "This script updates imports after running migrate_structure.sh"
        exit 1
    fi
    
    case "$phase" in
        phase1)
            update_imports_phase1
            ;;
        phase2)
            update_imports_phase2
            ;;
        *)
            log_warn "Invalid phase: $phase"
            exit 1
            ;;
    esac
    
    log_info "Import update completed!"
    log_warn "Please review the changes and test the app"
}

main "$@"

