# X-App-SDK Integration Guide

## Tổng quan

Tài liệu này mô tả cách tích hợp mini app với `x-app-sdk` và cách Flutter web wrapper (`XAppSDKService`) expose đầy đủ API như `getToken()`, `closeApp()`, `configUIApp()` cùng các tiện ích khác từ Super App.

## Cài đặt

### 1. Install x-app-sdk

```bash
npm install x-app-sdk@^1.1.2
```

### 2. Build Flutter web app

```bash
flutter build web
```

## Cách sử dụng

### Trong Super App

Super App không cần làm gì đặc biệt. x-app-sdk sẽ tự động detect Super App environment và sử dụng các method có sẵn.

**Lưu ý**: Mini app sử dụng x-app-sdk thật từ npm package, không phải mock.

### Trong Mini App (Flutter)

Mini app sẽ tự động lấy token từ Super App khi khởi động:

```dart
// Token được lấy tự động trong splash screen
// Không cần gọi thủ công

// Đóng app và trả về Super App
webCloseApp({
  'message': 'Task completed',
  'timestamp': DateTime.now().millisecondsSinceEpoch,
});
```

## API Reference

### Web Helper Functions

**Init & Diagnostics**
- `webInitializeXAppSDK()`: Khởi tạo x-app-sdk service
- `webIsSDKInitialized()`: Kiểm tra SDK đã khởi tạo chưa
- `webGetToken()`: Lấy token từ Super App
- `webGetCachedToken()`: Lấy token đã cache
- `webGetLastError()`: Lấy error message cuối cùng
- `webClearTokenCache()`: Xóa token cache
- `webResetSDK()`: Reset SDK service
- `webCloseApp(data) -> bool`: Đóng app, trả về `true` nếu host xử lý thành công; `false` khi chạy browser mode (dùng fallback điều hướng tại Flutter)

**Config**
- `webConfigUIApp(config)`: Thiết lập UI trong Super App

```dart
import 'package:flutter/foundation.dart';
import 'package:mypoint_flutter_app/web/web_helper.dart';

Future<void> configureHeader() async {
  if (!webIsSDKInitialized()) {
    await webInitializeXAppSDK();
  }

  final response = await webConfigUIApp({
    'headerTitle': 'Tên ứng dụng',
    'headerColor': '#ffffff',
    'headerTextColor': '#000000',
  });

  if (response != null) {
    debugPrint('Cấu hình thành công: $response');
  } else {
    debugPrint('Cấu hình thất bại: ${webGetLastError()}');
  }
}
```

**Device**
- `webCallPhone(phone)` / `webCall(phone)`: Gọi điện
- `webSendSms(phone)` / `webSms(phone)`: Mở app SMS
- `webVibrate()`: Rung thiết bị

**Location**
- `webCurrentLocation()`: Lấy vị trí hiện tại
- `webRequestLocationPermission()`: Xin quyền vị trí

**Media**
- `webOpenPickerImage(type)`: Mở image picker
- `webOpenPickerFile([options])`: Mở file picker

**Notification & Payment**
- `webListenNotificationEvent(onEvent)`: Lắng nghe notification
- `webPaymentRequest(payload)`: Gửi yêu cầu thanh toán
- `webListenPaymentEvent(onEvent)`: Lắng nghe sự kiện thanh toán

**Permission**
- `webPermissionsRequest(type)` / `webPremissionsRequest(type)`: Xin quyền theo SDK

**Store**
- `webSaveStore(data)`: Lưu dữ liệu
- `webGetStore()`: Lấy dữ liệu
- `webClearStore()`: Xóa dữ liệu

**User**
- `webGetInfo(key)`: Lấy thông tin user

### XAppSDKService

```dart
final service = XAppSDKService();

// Khởi tạo
await service.initialize();

// Lấy token
String? token = await service.getToken();

// Đóng app
final closed = await service.closeApp({'message': 'Done'});
if (!closed) {
  // Không có Super App host => fallback
  Navigator.of(context).pushReplacementNamed(onboardingRoute);
}

// Kiểm tra trạng thái
bool isReady = service.isInitialized;
String? cachedToken = service.cachedToken;
String? error = service.lastError;

// Ví dụ lắng nghe notification
final removeNotification = await service.listenNotificationEvent((event) {
  debugPrint('Notification event: $event');
});
```

## Luồng hoạt động

1. **Khởi tạo**: App khởi tạo x-app-sdk khi start
2. **Splash Screen**: Tự động gọi `getToken()` để lấy token
3. **Fallback**: Nếu SDK không có token, fallback về URL params
4. **Login**: Sử dụng token để đăng nhập
5. **Close App**: Khi cần đóng app, gọi `closeApp()`

## Test

### Local Development

```bash
# Test với mock SDK
./scripts/test_x_app_sdk.sh
```

### Production

Không cần thay đổi gì. x-app-sdk sẽ tự động detect Super App environment và hoạt động đúng.

**Lưu ý**: x-app-sdk được load từ npm package, tự động detect Super App environment.

## Troubleshooting

### SDK không khởi tạo được

- Kiểm tra console log: `❌ XAppSDKService: x-app-sdk not found in window`
- Đảm bảo Super App đã load x-app-sdk trước khi load mini app

### Token không lấy được

- Kiểm tra console log: `❌ SplashScreen - Failed to get token from SDK`
- Fallback sẽ tự động sử dụng URL params
- Kiểm tra Super App có expose `getToken()` method không

### CloseApp không hoạt động

- Kiểm tra console log: `❌ XAppSDKService: closeApp method not found`
- Fallback sẽ tự động sử dụng `window.history.back()` hoặc `window.close()`

## Files liên quan

- `lib/web/x_app_sdk_service.dart` - Service chính
- `lib/web/web_helper_web.dart` - Web implementation
- `lib/web/web_helper_stub.dart` - Stub cho non-web platforms
- `lib/web/web_helper.dart` - Export file
- `lib/screen/splash/splash_screen_viewmodel.dart` - Tích hợp trong splash
- `lib/base/app_navigator.dart` - Sử dụng closeApp
- `lib/core/app_initializer.dart` - Khởi tạo SDK
- `web/index.html` - Mock SDK cho development
- `scripts/test_x_app_sdk.sh` - Script test
