#!/usr/bin/env bash

# Build and package the Flutter web app for PRODUCTION delivery with optimizations.

set -euo pipefail

PROJECT_ROOT="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
TIMESTAMP="$(date +%Y%m%d_%H%M%S)"
OUT_DIR="web_prod_export_${TIMESTAMP}"
ZIP_FILE="${OUT_DIR}.zip"

cd "${PROJECT_ROOT}"

write_prod_env() {
  cat > assets/config/env.json <<'EOF'
{
  "flavor":"pro",
  "baseUrl":"https://api.mypoint.com.vn/8854/gup2start/rest",
  "t3Token":"runner-env-flavor-pro",
  "enableLogging":false
}
EOF
  echo "📋 Production config written to assets/config/env.json"
}

copy_x_app_sdk() {
  local src="node_modules/x-app-sdk/dist/index.es.js"
  local dest="build/web/js/x_app_sdk_bundle.js"
  if [ ! -f "${src}" ]; then
    echo "❌ x-app-sdk bundle not found at ${src}" >&2
    exit 1
  fi
  mkdir -p "$(dirname "${dest}")"
  cp "${src}" "${dest}"
  echo "✅ Copied x-app-sdk bundle."
}

compress_assets() {
  local dir="$1"
  if command -v gzip >/dev/null 2>&1; then
    find "${dir}" -type f \( -name '*.js' -o -name '*.css' -o -name '*.json' -o -name '*.wasm' -o -name '*.svg' -o -name '*.html' \) -exec gzip -9 -kf {} \;
  else
    echo "⚠️ gzip not available, skipping .gz artifacts"
  fi
  if command -v brotli >/dev/null 2>&1; then
    find "${dir}" -type f \( -name '*.js' -o -name '*.css' -o -name '*.json' -o -name '*.wasm' -o -name '*.svg' -o -name '*.html' \) -exec brotli -f -k -q 11 {} \;
  else
    echo "⚠️ brotli not available, skipping .br artifacts"
  fi
}

echo "🚀 Building PRODUCTION export (optimized)..."
write_prod_env

echo "🧹 Cleaning previous artifacts..."
flutter clean || true
rm -rf .dart_tool build || true

echo "📦 Getting Flutter packages..."
flutter pub get

echo "🔨 Flutter build web (release, CanvasKit, no source maps)..."
flutter build web \
  --release \
  --no-source-maps \
  --pwa-strategy=none \
  --dart-define=FLUTTER_WEB_USE_SKIA=true \
  --dart-define=FLUTTER_WEB_USE_SKWASM=false \
  --no-wasm-dry-run

copy_x_app_sdk

echo "📁 Preparing export directory: ${OUT_DIR}"
mkdir -p "${OUT_DIR}"
cp -r build/web/* "${OUT_DIR}/"

if [ -f "web/firebase-messaging-sw.js" ] && [ ! -f "${OUT_DIR}/firebase-messaging-sw.js" ]; then
  cp web/firebase-messaging-sw.js "${OUT_DIR}/"
fi

# Helper script for local preview with SPA fallback
cat > "${OUT_DIR}/serve_local.sh" <<'EOF'
#!/usr/bin/env bash

PORT="${1:-8080}"

python3 - "$PORT" <<'PY'
import http.server
import socketserver
import os
import sys

ROOT = os.getcwd()
PORT = int(sys.argv[1])

class SPAHandler(http.server.SimpleHTTPRequestHandler):
    def do_GET(self):
        resolved = self.translate_path(self.path)
        if not os.path.exists(resolved):
            self.path = '/index.html'
        return super().do_GET()

    def log_message(self, fmt, *args):
        print(f"🌐 {fmt % args}")

with socketserver.TCPServer(('', PORT), SPAHandler) as httpd:
    print(f"🚀 Serving {ROOT} at http://localhost:{PORT}")
    print("💡 SPA fallback enabled. Ctrl+C to stop.")
    httpd.serve_forever()
PY
EOF
chmod +x "${OUT_DIR}/serve_local.sh"

if [ "${KEEP_CANVASKIT:-0}" != "1" ]; then
  echo "🧹 Removing CanvasKit bundle to shrink export (set KEEP_CANVASKIT=1 to keep)..."
  rm -rf "${OUT_DIR}/canvaskit"
else
  echo "ℹ️ KEEP_CANVASKIT=1 → giữ nguyên thư mục canvaskit."
fi

echo "🗜️ Precompressing assets..."
compress_assets "${OUT_DIR}"

echo "📦 Creating zip archive ${ZIP_FILE}..."
zip -rq "${ZIP_FILE}" "${OUT_DIR}"

echo ""
echo "🎉 PRODUCTION export ready!"
echo "   Folder : ${OUT_DIR}"
echo "   Zip    : ${ZIP_FILE}"
echo ""
echo "📌 Deliverable: send the zip (or folder) to the hosting team. Serve with gzip/brotli if possible."
echo ""
echo "▶️ Quick preview command:"
echo "   cd ${OUT_DIR} && ./serve_local.sh 8080"
echo "⚠️  Avoid using 'python3 -m http.server' directly; it does not handle SPA routes and will 404 on deep links."
