import 'package:flutter/material.dart';
import 'package:mypoint_flutter_app/extensions/datetime_extensions.dart';
import '../../../extensions/date_format.dart';

class HistoryPointCashBackResponse {
  final double points;
  final List<HistoryPointCashBackOrderModel>? orders;

  HistoryPointCashBackResponse({required this.points, this.orders});

  factory HistoryPointCashBackResponse.fromJson(Map<String, dynamic> json) {
    return HistoryPointCashBackResponse(
      points: (json['points'] ?? 0).toDouble(),
      orders: (json['orders'] as List<dynamic>?)
          ?.map((e) => HistoryPointCashBackOrderModel.fromJson(e))
          .toList(),
    );
  }
}

class HistoryPointCashBackOrderModel {
  final String? status;
  final double? orderAtTime;
  final String? code, brand, logo;
  final double? price;
  final int id, points;
  final List<ProductRefundModel> products;

  HistoryPointCashBackOrderModel({
    required this.status,
    required this.orderAtTime,
    required this.code,
    required this.brand,
    required this.logo,
    required this.price,
    required this.id,
    required this.points,
    required this.products,
  });

  String get timeShow {
    if (orderAtTime == null) return '';
    final date = DateTime.fromMillisecondsSinceEpoch(((orderAtTime ?? 0) * 1000).toInt());
    return date.toFormattedString();
  }

  factory HistoryPointCashBackOrderModel.fromJson(Map<String, dynamic> json) {
    return HistoryPointCashBackOrderModel(
      status: json['status'],
      orderAtTime: (json['order_at_time'] as num?)?.toDouble(),
      code: json['code'],
      brand: json['brand'],
      logo: json['logo'],
      price: (json['price'] as num?)?.toDouble(),
      id: json['id'],
      points: json['points'],
      products: (json['products'] as List<dynamic>?)
          ?.map((e) => ProductRefundModel.fromJson(e))
          .toList() ??
          [],
    );
  }

  int get productQuantity => products.fold(0, (sum, item) => sum + item.quantity);
}

class ProductRefundModel {
  final String? name, logo, brand;
  final String code;
  final double price;
  final int quantity, points;
  final String? status;

  ProductRefundModel({
    required this.name,
    required this.logo,
    required this.brand,
    required this.code,
    required this.price,
    required this.quantity,
    required this.points,
    required this.status,
  });

  factory ProductRefundModel.fromJson(Map<String, dynamic> json) {
    return ProductRefundModel(
      name: json['name'],
      logo: json['logo'],
      brand: json['brand'],
      code: json['code'],
      price: (json['price'] as num).toDouble(),
      quantity: json['quantity'],
      points: json['points'],
      status: json['status'],
    );
  }
}

enum CashBackPointOrderStatus {
  pending,
  approved,
  confirmed,
  reject,
}

extension OrderStatusX on CashBackPointOrderStatus {
  String get tag {
    switch (this) {
      case CashBackPointOrderStatus.pending:
        return "Chờ xử lý";
      case CashBackPointOrderStatus.approved:
        return "Tạm duyệt";
      case CashBackPointOrderStatus.confirmed:
        return "Đã hoàn";
      case CashBackPointOrderStatus.reject:
        return "Đã huỷ";
    }
  }

  String get rawValue {
    switch (this) {
      case CashBackPointOrderStatus.pending:
        return "PENDING";
      case CashBackPointOrderStatus.approved:
        return "APPROVED";
      case CashBackPointOrderStatus.confirmed:
        return "CONFIRMED";
      case CashBackPointOrderStatus.reject:
        return "REJECTED";
    }
  }

  Color get color {
    switch (this) {
      case CashBackPointOrderStatus.pending:
        return Colors.orange;
      case CashBackPointOrderStatus.approved:
        return Colors.yellow.shade900;
      case CashBackPointOrderStatus.confirmed:
        return Colors.green;
      case CashBackPointOrderStatus.reject:
        return Colors.red;
    }
  }
}