import 'dart:ffi' as MediaQuery;
import 'dart:ui' as ui;
import 'package:device_info_plus/device_info_plus.dart';
import 'package:flutter/foundation.dart';
import 'package:package_info_plus/package_info_plus.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:uuid/uuid.dart';
import 'dart:io' show Platform;

class DeviceDetails {
  final String? hardwareType; // "Mobile" | "Tablet" | "Desktop" | "Web"
  final String? hardwareModel; // iPhone15,3 | iPad... | Samsung SM-... | ...
  final String? operatingSystem; // "iOS" | "Android" | "macOS" | ...
  final String? osVersion; // "17.6" | "14 (SDK 34)" | ...
  final String? userDeviceName; // iOS: tên máy; Android: fallback
  final String? appVersion; // build-name, ví dụ "3.2.6"
  final String? appBuildNumber; // build-number, ví dụ "326"

  const DeviceDetails({
    this.hardwareType,
    this.hardwareModel,
    this.operatingSystem,
    this.osVersion,
    this.userDeviceName,
    this.appVersion,
    this.appBuildNumber,
  });

  Map<String, String> toMap() => {
    'hardware_type': hardwareType ?? 'Unknown',
    'hardware_model': hardwareModel ?? 'Unknown',
    'operating_system': operatingSystem ?? 'Unknown',
    'os_version': osVersion ?? 'Unknown',
    'user_device_name': userDeviceName ?? 'Unknown',
    'app_version': appVersion ?? 'Unknown',
    'app_build_number': appBuildNumber ?? 'Unknown',
  };

  @override
  String toString() => toMap().toString();
}

class DeviceInfo {
  static Future<String> getDeviceId() async {
    SharedPreferences prefs = await SharedPreferences.getInstance();
    String? deviceId = prefs.getString('device_id');
    if (deviceId == null) {
      deviceId = const Uuid().v4(); // Tạo UUID mới
      await prefs.setString('device_id', deviceId);
    }
    return deviceId;
  }

  static Future<DeviceDetails> getDetails() async {
    final deviceInfo = DeviceInfoPlugin();
    final pkg = await PackageInfo.fromPlatform();
    // OS + version
    String os, osVersion = '';
    if (kIsWeb) {
      os = 'Web';
      final web = await deviceInfo.webBrowserInfo;
      osVersion = [
        web.browserName.name,
        if ((web.appVersion ?? '').isNotEmpty) '(${web.appVersion})',
      ].where((e) => e.toString().isNotEmpty).join(' ');
    } else if (Platform.isIOS) {
      os = 'iOS';
      final ios = await deviceInfo.iosInfo;
      osVersion = ios.systemVersion;
    } else if (Platform.isAndroid) {
      os = 'Android';
      final and = await deviceInfo.androidInfo;
      final rel = and.version.release;
      final sdk = and.version.sdkInt.toString();
      osVersion = sdk.isEmpty ? rel : '$rel (SDK $sdk)';
    } else {
      os = Platform.operatingSystem; // macOS/linux/windows
      osVersion = Platform.operatingSystemVersion;
    }
    // Model + userDeviceName + iPad fallback
    String hardwareModel = 'Unknown';
    String userDeviceName = '';
    bool fallbackIsTablet = false;

    if (kIsWeb) {
      hardwareModel = 'Browser';
    } else if (Platform.isIOS) {
      final ios = await deviceInfo.iosInfo;
      hardwareModel = ios.utsname.machine;
      userDeviceName = ios.name;
      fallbackIsTablet =
          (ios.model.toLowerCase().contains('ipad')) || (ios.utsname.machine.toLowerCase().startsWith('ipad'));
    } else if (Platform.isAndroid) {
      final and = await deviceInfo.androidInfo;
      final brand = and.brand.trim();
      final model = and.model.trim();
      hardwareModel = [brand, model].where((e) => e.isNotEmpty).join(' ');
      userDeviceName = and.device;
      fallbackIsTablet = false;
    }

    final hardwareType = _detectHardwareTypeWithoutContext(os: os, fallbackIsTablet: fallbackIsTablet);

    return DeviceDetails(
      hardwareType: hardwareType,
      hardwareModel: hardwareModel,
      operatingSystem: os,
      osVersion: osVersion,
      userDeviceName: userDeviceName,
      appVersion: pkg.version,
      appBuildNumber: pkg.buildNumber,
    );
  }

  /// Không cần BuildContext: dùng kích thước của view đầu tiên từ PlatformDispatcher.
  /// Quy ước: shortestSide >= 600 logical pixels => Tablet.
  static String _detectHardwareTypeWithoutContext({required String os, required bool fallbackIsTablet}) {
    if (os == 'Web') return 'Web';
    if (os != 'iOS' && os != 'Android') return 'Desktop';

    try {
      final views = ui.PlatformDispatcher.instance.views;
      final view = views.isNotEmpty ? views.first : ui.PlatformDispatcher.instance.implicitView;
      if (view != null) {
        final logicalSize = view.physicalSize / view.devicePixelRatio;
        final shortest = logicalSize.shortestSide;
        return shortest >= 600 ? 'Tablet' : 'Mobile';
      }
    } catch (_) {
      // ignore and use fallback
    }
    return fallbackIsTablet ? 'Tablet' : 'Mobile';
  }
}
