import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:local_auth/local_auth.dart';

enum BiometricTypeEnum {
  none,
  fingerprint,
  faceId,
}

class BiometricManager {
  final LocalAuthentication _localAuth = LocalAuthentication();

  /// Kiểm tra xem thiết bị hỗ trợ loại sinh trắc học nào: faceID, fingerprint, none
  Future<BiometricTypeEnum> checkDeviceBiometric() async {
    try {
      final availableBiometrics = await _localAuth.getAvailableBiometrics();
      if (availableBiometrics.contains(BiometricType.face)) {
        return BiometricTypeEnum.faceId;
      } else if (availableBiometrics.contains(BiometricType.fingerprint)) {
        return BiometricTypeEnum.fingerprint;
      }
      return BiometricTypeEnum.none;
    } catch (e) {
      debugPrint("Lỗi checkDeviceBiometric: $e");
      return BiometricTypeEnum.none;
    }
  }

  Future<bool> canCheckBiometrics() async {
    try {
      final canCheck = await _localAuth.canCheckBiometrics;
      final isSupported = await _localAuth.isDeviceSupported();
      return canCheck && isSupported;
    } catch (e) {
      debugPrint("Lỗi canCheckBiometrics/isDeviceSupported: $e");
      return false;
    }
  }

  /// Thực hiện xác thực bằng sinh trắc
  ///  - `localizedReason` là chuỗi yêu cầu xác thực hiển thị mặc định trên hệ thống
  ///  - Trả về true nếu user xác thực thành công, false nếu user huỷ hoặc thất bại
  Future<bool> authenticateBiometric({String localizedReason = "Xác thực để đăng nhập"}) async {
    try {
      final didAuthenticate = await _localAuth.authenticate(
        localizedReason: localizedReason,
        options: const AuthenticationOptions(biometricOnly: true),
      );
      return didAuthenticate;
    } catch (e) {
      debugPrint("Lỗi authenticateBiometric: $e");
      return false;
    }
  }

  /// (Tuỳ chọn) Hiển thị trước một dialog hỏi "Có muốn xác thực bằng vân tay/FaceID hay không?"
  /// Nếu user bấm "Đồng ý", mới gọi authenticateBiometric
  Future<bool> showCustomBiometricDialog(
      BuildContext context, {
        String title = "Sử dụng sinh trắc học",
        String content = "Bạn có muốn đăng nhập bằng vân tay/Face ID không?",
        String confirmText = "Đồng ý",
        String cancelText = "Huỷ",
      }) async {
    final result = await Get.dialog<bool>(
      AlertDialog(
        title: Text(title),
        content: Text(content),
        actions: [
          TextButton(
            onPressed: () => Get.back(result: false),
            child: Text(cancelText),
          ),
          TextButton(
            onPressed: () => Get.back(result: true),
            child: Text(confirmText),
          ),
        ],
      ),
    );
    if (result == true) {
      // Chỉ khi user chọn Đồng ý thì mới gọi authenticateBiometric
      return await authenticateBiometric();
    }
    // Người dùng huỷ dialog => false
    return false;
  }
}
