import 'dart:async';
import 'package:flutter/foundation.dart';
import 'package:get/get.dart';
import 'package:mypoint_flutter_app/app/routing/app_navigator.dart';
import 'package:mypoint_flutter_app/core/utils/extensions/string_extension.dart';
import 'package:mypoint_flutter_app/core/network/restful_api_viewmodel.dart';
import 'package:mypoint_flutter_app/core/network/restful_api_client_all_request.dart';
import 'package:mypoint_flutter_app/features/register_campaign/model/registration_form_package_model.dart';
import 'package:mypoint_flutter_app/shared/router_gage.dart';
import '../../core/services/web/web_info_data.dart';
import '../../core/services/web/x_app_sdk_service.dart';
import '../login/model/login_token_response_model.dart';
import '../personal/model/profile_response_model.dart';
import 'models/update_response_model.dart';
import '../../shared/preferences/data_preference.dart';
import '../../shared/preferences/point/point_manager.dart';
import 'package:url_launcher/url_launcher.dart';
import '../../core/services/web/web_helper.dart';
import '../popup_manager/popup_manager_viewmodel.dart';

class SplashScreenViewModel extends RestfulApiViewModel {
  void Function(UpdateResponseModel? data)? checkUpdateResponse;
  var _updateLink = '';
  static const Duration _networkTimeout = Duration(seconds: 20);
  static const Duration _sdkTimeout = Duration(seconds: 10);

  Future<void> checkUpdateApp() async {
    debugPrint('🔍 SplashScreen - Checking for app update... ${DateTime.now().toString()}');
    if (kIsWeb) {
      checkUpdateResponse?.call(null);
      return;
    }
    try {
      final response = await client.checkUpdateApp().timeout(_networkTimeout);
      _updateLink = response.data?.updateRequest?.firstOrNull?.updateLink.orEmpty ?? '';
      checkUpdateResponse?.call(response.data);
    } on Object catch (e) {
      debugPrint('⚠️ SplashScreen - checkUpdateApp failed: $e');
      checkUpdateResponse?.call(null);
    }
  }

  Future<void> openLink() async {
    if (_updateLink.isEmpty) return;
    final Uri url = Uri.parse(_updateLink);
    if (await canLaunchUrl(url)) {
      await launchUrl(url);
    }
  }

  Future<void> makeDataFollowInitApp() async {
    try {
      if (kIsWeb) {
        final token = await _getTokenFromSDK();
        if (token.orEmpty.isNotEmpty) {
          await DataPreference.instance.saveLoginToken(LoginTokenResponseModel(accessToken: token));
          await _cacheProfileFromSDK();
        }
      }
      if (DataPreference.instance.logged) {
        await _loadProfileAndNavigate();
      } else {
        directionWhenTokenInvalid();
      }
    } on Object catch (e) {
      debugPrint('❌ SplashScreen - makeDataFollowInitApp error: $e');
      DataPreference.instance.clearLoginToken();
      directionWhenTokenInvalid();
    }
  }

  /// Get token from x-app-sdk (web only)
  Future<String?> _getTokenFromSDK() async {
    if (!kIsWeb) {
      debugPrint('🔍 SplashScreen - Not on web, skipping SDK token retrieval');
      return null;
    }
    try {
      debugPrint('🔍 SplashScreen - Attempting to get token from x-app-sdk...');
      await webInitializeXAppSDK().timeout(_sdkTimeout);
      if (!webIsSDKInitialized()) {
        debugPrint('⚠️ SplashScreen - SDK not initialized, skipping');
        return null;
      }
      // Get token from SDK
      final token = await webGetToken().timeout(_sdkTimeout);
      if (token != null && token.isNotEmpty) {
        debugPrint('✅ SplashScreen - Token retrieved from x-app-sdk: $token...');
        return token;
      } else {
        final error = webGetLastError();
        debugPrint('❌ SplashScreen - Failed to get token from SDK: $error');
        return null;
      }
    } catch (e) {
      debugPrint('❌ SplashScreen - Error getting token from SDK: $e');
      return null;
    }
  }

  Future<void> _cacheProfileFromSDK() async {
    if (!kIsWeb) return;
    try {
      final sdkProfile = await WebData.getInfoFromSDK(forceRefresh: true);
      if (sdkProfile == null || sdkProfile.isEmpty) {
        debugPrint('⚠️ SplashScreen - No profile data from x-app-sdk');
        return;
      }
      debugPrint('✅ SplashScreen - Cached user profile from x-app-sdk: $sdkProfile');
      final response = client.webUpdateProfile({'metadata': sdkProfile.toJsonString()});
      debugPrint('🔍 SplashScreen - webUpdateProfile response: ${response.toString()}');
    } catch (e) {
      debugPrint('⚠️ SplashScreen - Failed to cache SDK profile: $e');
    }
  }

  Future<void> _loadProfileAndNavigate() async {
    final profile = await _fetchUserProfile();
    if (profile == null) {
      DataPreference.instance.clearLoginToken();
      directionWhenTokenInvalid();
      return;
    }
    await _prepareInitialData(profile);
    _goToMain();
  }

  Future<void> directionWhenTokenInvalid() async {
    if (kIsWeb && !XAppSDKService.isBrowserMode()) {
      debugPrint('❌ No token found on web platform. directionWhenTokenInvalid');
      Future.microtask(() {
        AppNavigator.showAlertError(
            content: "MyPoint cần thông tin của bạn.\nSố điện thoại được sử dụng để định danh và sử dụng các tính năng trên MyPoint.",
            onConfirmed: () async {
              final closeSuccess = await webCloseApp({
                'message': 'User acknowledged missing token',
                'timestamp': DateTime
                    .now()
                    .millisecondsSinceEpoch,
              });
              debugPrint('❌ No token found on web, user acknowledged. Close app success: $closeSuccess');
              if (!closeSuccess) {
                Get.offAllNamed(onboardingScreen);
              }
            }
        );
      });
      return;
    }
    final phone = DataPreference.instance.phoneNumberUsedForLoginScreen;
    final displayName = DataPreference.instance.displayName;
    if (phone.isEmpty) {
      Get.offAllNamed(onboardingScreen);
    } else {
      Get.offAllNamed(loginScreen, arguments: {"phone": phone, 'fullName': displayName});
    }
  }

  Future<ProfileResponseModel?> _fetchUserProfile() async {
    try {
      final response = await client.getUserProfile().timeout(_networkTimeout);
      if (response.isSuccess) {
        return response.data;
      }
      debugPrint('⚠️ SplashScreen - getUserProfile failed: ${response.errorMessage ?? response.message}');
    } on TimeoutException catch (_) {
      debugPrint('⚠️ SplashScreen - getUserProfile timeout');
    } catch (e) {
      debugPrint('⚠️ SplashScreen - getUserProfile error: $e');
    }
    return null;
  }

  Future<void> _prepareInitialData(ProfileResponseModel profile) async {
    await _runSafely(() => DataPreference.instance.saveUserProfile(profile), 'cache user profile');
    await _runSafely(() => UserPointManager().fetchUserPoint(), 'fetch user point');
    await _runSafely(() => PopupManagerViewModel.instance.ensureLoaded(), 'preload popups');
  }

  Future<void> _runSafely(Future<void> Function() task, String label) async {
    try {
      await task().timeout(_networkTimeout);
    } on TimeoutException catch (_) {
      debugPrint('⚠️ SplashScreen - $label timeout');
    } catch (e) {
      debugPrint('⚠️ SplashScreen - $label failed: $e');
    }
  }

  void _goToMain() {
    if (Get.currentRoute == mainScreen) return;
    if (kIsWeb || Get.currentRoute == onboardingScreen) {
      Get.offNamed(mainScreen);
    } else {
      Get.offAllNamed(mainScreen);
    }
  }
}
