import 'package:flutter/material.dart';
import 'package:flutter_widget_from_html_core/flutter_widget_from_html_core.dart';
import 'package:get/get.dart';
import '../../../core/theme/base_color.dart';
import '../image_loader.dart';
import 'data_alert_model.dart';

enum ButtonsDirection { row, column }

class CustomAlertDialog extends StatelessWidget {
  final DataAlertModel alertData;
  final ButtonsDirection direction;
  final bool showCloseButton;

  const CustomAlertDialog({
    super.key,
    required this.alertData,
    this.direction = ButtonsDirection.column,
    this.showCloseButton = true,
  });

  @override
  Widget build(BuildContext context) {
    return Dialog(
      shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(16)),
      child: Container(
        padding: const EdgeInsets.all(0),
        decoration: BoxDecoration(borderRadius: BorderRadius.circular(16), color: Colors.white),
        child: Stack(
          children: [
            Column(
              mainAxisSize: MainAxisSize.min,
              children: [
                _buildHeaderImage(),
                Padding(
                  padding: const EdgeInsets.all(8),
                  child: Column(
                    children: [
                      if ((alertData.title ?? "").isNotEmpty)... [
                        Text(
                          alertData.title!,
                          style: const TextStyle(fontSize: 18, fontWeight: FontWeight.bold),
                          textAlign: TextAlign.center,
                        ),
                        const SizedBox(height: 4)
                      ],
                      if (alertData.description != null)... [
                        HtmlWidget('''
                    <div style="text-align: center;">
                      ${alertData.description!}
                    </div>
                    '''),
                        const SizedBox(height: 4),
                      ],
                      if ((alertData.content ?? "").isNotEmpty)... [
                        Text(
                          alertData.content!,
                          style: TextStyle(fontSize: 14, fontWeight: FontWeight.w600, color: BaseColor.primary500),
                          textAlign: TextAlign.center,
                        ),
                        const SizedBox(height: 4),
                      ],
                      _buildButtons(),
                    ],
                  ),
                ),
              ],
            ),
            // Close Button (X) ở góc phải trên
            if (showCloseButton)
              Positioned(
                top: 8,
                right: 8,
                child: GestureDetector(
                  onTap: () => Get.back(),
                  child: const Icon(Icons.close, color: Colors.black, size: 24),
                ),
              ),
          ],
        ),
      ),
    );
  }

  Widget _buildHeaderImage() {
    if ((alertData.urlHeaderImage ?? "").isNotEmpty) {
      return ClipRRect(
        borderRadius: BorderRadius.circular(12),
        child: Container(
          // color: Colors.grey,
          child: loadNetworkImage(
            url: alertData.urlHeaderImage,
            fit: BoxFit.fill,
            placeholderAsset: "assets/images/ic_pipi_03.png",
          ),
        ),
      );
    }
    final localHeaderImage = (alertData.localHeaderImage ?? "");
    final localImage = localHeaderImage.isNotEmpty ? localHeaderImage : "assets/images/ic_pipi_03.png";
    return ClipRRect(borderRadius: BorderRadius.circular(12), child: Image.asset(localImage));
  }

  Widget _buildButtons() {
    return direction == ButtonsDirection.column
        ? Column(
          children:
              alertData.buttons
                  ?.whereType<AlertButton>()
                  .map(
                    (btn) => Padding(
                      padding: const EdgeInsets.symmetric(vertical: 4),
                      child: ElevatedButton(
                        style: ElevatedButton.styleFrom(
                          backgroundColor: btn.bgColor,
                          shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(24)),
                          minimumSize: const Size(double.infinity, 48),
                        ),
                        onPressed: btn.onPressed,
                        child: Text(
                          btn.text,
                          style: TextStyle(color: btn.textColor, fontSize: 14, fontWeight: FontWeight.bold),
                        ),
                      ),
                    ),
                  )
                  .toList() ??
              [],
        )
        : Row(
          mainAxisAlignment: MainAxisAlignment.spaceEvenly,
          children:
              alertData.buttons
                  ?.whereType<AlertButton>()
                  .map(
                    (btn) => Expanded(
                      child: Padding(
                        padding: const EdgeInsets.symmetric(horizontal: 4),
                        child: ElevatedButton(
                          style: ElevatedButton.styleFrom(
                            backgroundColor: btn.bgColor,
                            shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(24)),
                            minimumSize: const Size(100, 48),
                          ),
                          onPressed: btn.onPressed,
                          child: Text(
                            btn.text,
                            style: TextStyle(color: btn.textColor, fontSize: 14, fontWeight: FontWeight.bold),
                          ),
                        ),
                      ),
                    ),
                  )
                  .toList() ??
              [],
        );
  }
}
