import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:local_auth/local_auth.dart';
import '../resources/base_color.dart';
import '../widgets/alert/custom_alert_dialog.dart';
import '../widgets/alert/data_alert_model.dart';

enum BiometricTypeEnum {
  none,
  fingerprint,
  faceId,
}

class BiometricManager {
  final LocalAuthentication _localAuth = LocalAuthentication();

  /// Kiểm tra xem thiết bị hỗ trợ loại sinh trắc học nào: faceID, fingerprint, none
  Future<BiometricTypeEnum> checkDeviceBiometric() async {
    try {
      final availableBiometrics = await _localAuth.getAvailableBiometrics();
      if (availableBiometrics.contains(BiometricType.face)) {
        return BiometricTypeEnum.faceId;
      } else if (availableBiometrics.contains(BiometricType.fingerprint)) {
        return BiometricTypeEnum.fingerprint;
      }
      return BiometricTypeEnum.none;
    } catch (e) {
      debugPrint("Lỗi checkDeviceBiometric: $e");
      return BiometricTypeEnum.none;
    }
  }

  Future<bool> canCheckBiometrics() async {
    try {
      final canCheck = await _localAuth.canCheckBiometrics;
      final isSupported = await _localAuth.isDeviceSupported();
      return canCheck && isSupported;
    } catch (e) {
      debugPrint("Lỗi canCheckBiometrics/isDeviceSupported: $e");
      return false;
    }
  }

  /// Thực hiện xác thực bằng sinh trắc
  ///  - `localizedReason` là chuỗi yêu cầu xác thực hiển thị mặc định trên hệ thống
  ///  - Trả về true nếu user xác thực thành công, false nếu user huỷ hoặc thất bại
  Future<bool> authenticateBiometric({String localizedReason = "Xác thực để đăng nhập"}) async {
    try {
      final didAuthenticate = await _localAuth.authenticate(
        localizedReason: localizedReason,
        options: const AuthenticationOptions(
          biometricOnly: false, // ✅ cho phép passcode hoặc vân tay / face ID
          stickyAuth: true,     // ✅ giữ lại phiên nếu app vào background
        ),
      );
      return didAuthenticate;
    } catch (e) {
      debugPrint("Lỗi authenticateBiometric: $e");
      return false;
    }
  }

  /// (Tuỳ chọn) Hiển thị trước một dialog hỏi "Có muốn xác thực bằng vân tay/FaceID hay không?"
  /// Nếu user bấm "Đồng ý", mới gọi authenticateBiometric
  Future<bool> showCustomBiometricDialog(
      BuildContext context, {
        String title = "Sử dụng sinh trắc học",
        String content = "Bạn có muốn đăng nhập bằng vân tay/Face ID không?",
      }) async {
    final result = await Get.dialog<bool>(
      CustomAlertDialog(
        alertData: DataAlertModel(
          localHeaderImage: "assets/images/bg_alert_header.png",
          title: title,
          description: content,
          buttons: [
            AlertButton(
              text: "Huỷ",
              onPressed: () => Get.back(result: false),
              bgColor: BaseColor.primary500,
              textColor: Colors.white,
            ),
            AlertButton(
              text: "Đồng ý",
              onPressed: () => Get.back(result: true),
              bgColor: BaseColor.primary500,
              textColor: Colors.white,
            ),
          ],
        ),
      ),
    );
    if (result == true) {
      // Chỉ khi user chọn Đồng ý thì mới gọi authenticateBiometric
      return await authenticateBiometric();
    }
    // Người dùng huỷ dialog => false
    return false;
  }
}
