import 'dart:convert';

/// Trả về Map<String, dynamic> từ payload của notification response.
/// - Hỗ trợ JSON thuần
/// - Hỗ trợ JSON bị URL-encode
/// - Fallback: parse kiểu querystring: a=1&b=2
Map<String, dynamic> parseNotificationPayload(String? payload) {
  if (payload == null) return {};
  String s = payload.trim();
  if (s.isEmpty) return {};

  // 1) Nếu là JSON thuần
  Map<String, dynamic>? tryJson(String text) {
    try {
      final v = jsonDecode(text);
      if (v is Map) {
        // ép key về String
        return v.map((k, val) => MapEntry(k.toString(), val));
      }
    } catch (_) {}
    return null;
  }

  // a) thử trực tiếp
  var m = tryJson(s);
  if (m != null) return m;

  // b) thử URL-decode rồi parse JSON lại
  final decoded = Uri.decodeFull(s);
  m = tryJson(decoded);
  if (m != null) return m;

  // 2) Fallback querystring: key1=val1&key2=val2
  try {
    // Bỏ ngoặc {..} nếu server lỡ bọc
    final q = s.startsWith('{') && s.endsWith('}') ? s.substring(1, s.length - 1) : s;
    final map = Uri.splitQueryString(q, encoding: utf8);
    return map.map((k, v) => MapEntry(k, v));
  } catch (_) {
    // 3) Fallback kiểu Dart Map.toString(): {a: 1, b: 2}
    try {
      final trimmed = s.trim();
      if (trimmed.startsWith('{') && trimmed.endsWith('}')) {
        final inner = trimmed.substring(1, trimmed.length - 1).trim();
        if (inner.isEmpty) return <String, dynamic>{};
        final Map<String, dynamic> out = {};
        // Tách theo dấu phẩy cấp 1 (không xử lý nested phức tạp)
        for (final pair in inner.split(',')) {
          final idx = pair.indexOf(':');
          if (idx <= 0) continue;
          final key = pair.substring(0, idx).trim();
          final val = pair.substring(idx + 1).trim();
          // Bỏ quote nếu có
          String normalize(String v) {
            if ((v.startsWith('"') && v.endsWith('"')) || (v.startsWith('\'') && v.endsWith('\''))) {
              return v.substring(1, v.length - 1);
            }
            return v;
          }
          final nk = normalize(key);
          final nv = normalize(val);
          out[nk] = nv;
        }
        if (out.isNotEmpty) return out;
      }
    } catch (_) {}
    // Cùng lắm trả string gốc
    return {'payload': s};
  }
}
