import 'package:flutter/foundation.dart';
import 'package:firebase_messaging/firebase_messaging.dart';

import '../../app/routing/directional_action_type.dart';
import '../../shared/navigation/directional_screen.dart';

enum NotificationType { fcm, moengage }

class NotificationRouter {
  static Future<void> handleRemoteMessage(RemoteMessage m) async {
      final data = PushNotification(info: m.data);
      handleDirectionNotification(data);
  }

  static Future<void> handleDirectionNotification(PushNotification notification) async {
    debugPrint('Parsed action type: ${notification.screenDirectional?.clickActionType}');
    debugPrint('Parsed action param: ${notification.screenDirectional?.clickActionParam}');
    notification.screenDirectional?.begin();
  }
}

class PushNotification {
  final Map<String, dynamic> info;

  PushNotification({required Map info})
      : info = info.map((k, v) => MapEntry(k.toString(), v));

  NotificationType get type =>
      info.containsKey('moengage') ? NotificationType.moengage : NotificationType.fcm;

  String get title {
    final rootTitle = _asString(info['title']) ?? _asString(info['gcm.notification.title']);
    if (rootTitle != null) return rootTitle;
    final aps = _asMap(info['aps']);
    final alert = _asMap(aps?['alert']);
    return _asString(alert?['title']) ?? '';
  }

  String get body {
    final rootBody = _asString(info['body']) ?? _asString(info['gcm.notification.body']);
    if (rootBody != null) return rootBody;
    final aps = _asMap(info['aps']);
    final alert = _asMap(aps?['alert']);
    return _asString(alert?['body']) ?? '';
  }

  String? get timeString => _asString(info['time']);

  String? get id => _asString(info['notification_id']);

  DirectionalScreen? get screenDirectional {
    debugPrint('=== PARSING NOTIFICATION DATA ===');
    debugPrint('Raw info: $info');
    
    String? name;
    String? param;

    final extra = _asMap(info['app_extra']);
    debugPrint('App extra: $extra');
    
    final screenData = _asMap(extra?['screenData']);
    debugPrint('Screen data: $screenData');
    
    if (screenData != null) {
      name = _asString(screenData[Defines.actionType]);
      param = _asString(screenData[Defines.actionParams]);
      debugPrint('From screenData - name: $name, param: $param');
    } else {
      name = _asString(info[Defines.actionType]);
      param = _asString(info[Defines.actionParams]);
      debugPrint('From info - name: $name, param: $param');
    }

    DirectionalScreen? screen;
    if (name != null || param != null) {
      debugPrint('Building DirectionalScreen with name: $name, param: $param');
      screen = DirectionalScreen.build(clickActionType: name, clickActionParam: param);
    }
    
    debugPrint('No action data found, using default notification screen with ID: $id');
    debugPrint('Title: $title, Body: $body');
    screen ??= DirectionalScreen.buildByName(name: DirectionalScreenName.notification, clickActionParam: id);
    screen?.extraData = {
      'title': title,
      'body': body,
    };
    return screen;
  }

  // ===== Helpers =====
  static String? _asString(dynamic v) => v is String ? v : null;

  static Map<String, dynamic>? _asMap(dynamic v) {
    if (v is Map) {
      // ép key về String để thao tác dễ hơn
      return v.map((k, val) => MapEntry(k.toString(), val));
    }
    return null;
  }
}