import 'package:flutter/material.dart';
import 'package:mypoint_flutter_app/app/routing/app_navigator.dart';
import 'package:mypoint_flutter_app/main.dart' show routeObserver;
import '../alert/custom_alert_dialog.dart';
import '../alert/data_alert_model.dart';
import '../alert/popup_data_model.dart';

abstract class BaseScreen extends StatefulWidget {
  const BaseScreen({super.key});
}

abstract class BaseState<Screen extends BaseScreen> extends State<Screen> with RouteAware {
  ModalRoute<dynamic>? _route;

  @override
  void initState() {
    super.initState();
    WidgetsBinding.instance.addPostFrameCallback((_) => onInit());
  }

  @override
  void dispose() {
    if (_route != null) {
      routeObserver.unsubscribe(this);
    }
    onDispose();
    super.dispose();
  }

  @override
  void didChangeDependencies() {
    super.didChangeDependencies();
    _setupRouteObserver();
  }

  void _setupRouteObserver() {
    final modalRoute = ModalRoute.of(context);
    if (modalRoute != null && modalRoute is PageRoute && modalRoute != _route) {
      _route = modalRoute;
      routeObserver.subscribe(this, modalRoute);
    }
  }

  // MARK: - Core Lifecycle Methods
  /// Called when the widget is first inserted into the tree.
  /// Use this to initialize data, setup listeners, etc.
  void onInit() {
    debugPrint("onInit: $runtimeType");
  }

  /// Called when the widget is removed from the tree.
  /// Use this to cleanup resources, cancel timers, etc.
  void onDispose() {
    debugPrint("onDispose: $runtimeType");
  }

  // MARK: - Route Visibility Methods
  /// Called when the route has become visible.
  /// Use this to start timers, refresh data, etc.
  void onRouteDidAppear() {
    debugPrint("onRouteDidAppear: $runtimeType");
  }

  /// Called when the route is about to be covered or popped.
  /// Use this to pause operations, save state, etc.
  void onRouteWillDisappear() {
    debugPrint("onRouteWillDisappear: $runtimeType");
  }

  // MARK: - UI Helper Methods
  /// Shows a popup dialog with custom data
  void showPopup({
    required PopupDataModel data,
    bool? barrierDismissible,
    bool showCloseButton = false,
    ButtonsDirection direction = ButtonsDirection.column,
  }) {
    showAlert(
      data: data.dataAlertModel,
      barrierDismissible: barrierDismissible ?? true,
      showCloseButton: showCloseButton,
      direction: direction,
    );
  }

  /// Shows an alert dialog with custom data
  void showAlert({
    required DataAlertModel data,
    bool? barrierDismissible,
    bool showCloseButton = true,
    ButtonsDirection direction = ButtonsDirection.column,
  }) {
    AppNavigator.showAlert(
      data: data,
      barrierDismissible: barrierDismissible,
      showCloseButton: showCloseButton,
      direction: direction,
    );
  }

  /// Shows an error alert with default styling
  void showAlertError({
    required String content,
    bool? barrierDismissible,
    String headerImage = "assets/images/ic_pipi_03.png",
    bool showCloseButton = true,
    VoidCallback? onConfirmed,
  }) {
    AppNavigator.showAlertError(
      content: content,
      barrierDismissible: barrierDismissible,
      headerImage: headerImage,
      showCloseButton: showCloseButton,
      onConfirmed: onConfirmed,
    );
  }

  /// Hides the keyboard
  void hideKeyboard() {
    FocusScope.of(context).unfocus();
  }

  // MARK: - RouteAware Implementation
  @override
  void didPush() {
    WidgetsBinding.instance.addPostFrameCallback((_) => onRouteDidAppear());
  }

  @override
  void didPopNext() => WidgetsBinding.instance.addPostFrameCallback((_) => onRouteDidAppear());

  @override
  void didPushNext() => _handleRouteDisappear();

  @override
  void didPop() => _handleRouteDisappear();

  void _handleRouteDisappear() {
    onRouteWillDisappear();
  }
}
