import 'package:flutter/material.dart';
import 'package:flutter/foundation.dart';
import 'package:firebase_messaging/firebase_messaging.dart';
import 'package:flutter_localizations/flutter_localizations.dart';
import 'package:get/get.dart';
import 'package:mypoint_flutter_app/networking/app_navigator.dart';
import 'package:mypoint_flutter_app/preference/data_preference.dart';
import 'package:mypoint_flutter_app/preference/point/point_manager.dart';
import 'package:mypoint_flutter_app/resources/base_color.dart';
import 'package:mypoint_flutter_app/screen/home/header_home_viewmodel.dart';
import 'package:mypoint_flutter_app/shared/router_gage.dart';
import 'base/app_loading.dart';
import 'env_loader.dart';
import 'networking/dio_http_service.dart';
import 'firebase/push_notification.dart';
import 'firebase/push_setup.dart';
import 'configs/url_params.dart';
import 'web/web_helper.dart';

final RouteObserver<PageRoute> routeObserver = RouteObserver<PageRoute>();
void main() async {
  WidgetsFlutterBinding.ensureInitialized();
  await loadEnv();
  await DataPreference.instance.init();
  DioHttpService();
  Get.put(HeaderThemeController(), permanent: true);
  // Web không dùng FCM: bỏ init Firebase Messaging để tránh SW/permission issues
  if (!kIsWeb) {
    await initFirebaseAndFcm();
  }
  // Chỉ fetch điểm khi đã đăng nhập, tránh 403 khi web chưa có token
  if (DataPreference.instance.logged) {
    await UserPointManager().fetchUserPoint();
  }
  // Đọc URL parameters cho web
  _handleWebUrlParams();
  runApp(const MyApp());
  WidgetsBinding.instance.addPostFrameCallback((_) {
    AppLoading().attach();
  });
  // Handle launch from notification when app was killed
  _handleInitialNotificationLaunch();
  // Handle launch from local notification tap when app was killed
  handleLocalNotificationLaunchIfAny();
}

void _handleWebUrlParams() {
  print('🔍 _handleWebUrlParams called, kIsWeb: $kIsWeb');
  if (!kIsWeb) return;
  final uri = Uri.base;
  print('🔍 Current URI: ${uri.toString()}');
  final token = uri.queryParameters['token'];
  final userId = uri.queryParameters['userId'];
  print('🔍 Web URL Params: {token: $token, user_id: $userId}');
  if (token != null && token.isNotEmpty) {
    UrlParams.setToken(token);
    UrlParams.setUserId(userId);
    print('✅ Token set from URL: $token');
    print('🔍 UrlParams after set: ${UrlParams.allParams}');
    // Clean URL to remove query params
    webReplaceUrl('/');
  } else {
    print('❌ No token found in URL parameters');
  }
}

class MyApp extends StatelessWidget {
  const MyApp({super.key});

  @override
  Widget build(BuildContext context) {
    return GetMaterialApp(
      navigatorKey: AppNavigator.key,
      navigatorObservers: [routeObserver],
      debugShowCheckedModeBanner: false,
      initialRoute: '/splash',
      theme: ThemeData(
        colorScheme: ColorScheme.fromSwatch(primarySwatch: Colors.brown),
        primaryColor: BaseColor.primary500,
      ),
      locale: const Locale('vi'),
      supportedLocales: const [
        Locale('vi', 'VN'), // Vietnamese
      ],
      localizationsDelegates: const [
        GlobalMaterialLocalizations.delegate,
        GlobalWidgetsLocalizations.delegate,
        GlobalCupertinoLocalizations.delegate,
      ],
      // home: SplashScreen(),
      getPages: RouterPage.pages(),
    );
  }
}

Future<void> _handleInitialNotificationLaunch() async {
  try {
    final initial = await FirebaseMessaging.instance.getInitialMessage();
    print('Checking initial message for app launch from terminated state...$initial');
    if (initial == null) return;
    WidgetsBinding.instance.addPostFrameCallback((_) {
      Future.delayed(const Duration(seconds: 1), () {
        NotificationRouter.handleRemoteMessage(initial);
      });
    });
  } catch (_) {}
}