# Giải thích chi tiết về Cấu trúc Mới

## Tổng quan

Cấu trúc mới được thiết kế theo **Clean Architecture** kết hợp với **Feature-based organization**, giúp code dễ maintain, scale và test hơn.

## Kiến trúc 3 Layers

### 1. Core Layer (`lib/core/`)

**Mục đích**: Chứa các thành phần được dùng chung trên toàn bộ app, không phụ thuộc vào business logic cụ thể.

#### `core/config/`
- **Chức năng**: Tất cả các configuration files
- **Files**:
  - `api_paths.dart` - API endpoints
  - `constants.dart` - App constants
  - `device_info.dart` - Device information
  - `env_loader.dart` - Environment loader
- **Lý do**: Tập trung tất cả configs ở một nơi, dễ quản lý và thay đổi

#### `core/initialization/`
- **Chức năng**: Khởi tạo app và các services
- **Files**:
  - `app_initializer.dart` - Main app initialization
  - `web_app_initializer.dart` - Web-specific initialization
  - `deep_link_service.dart` - Deep linking
- **Lý do**: Tách biệt logic khởi tạo, dễ test và maintain

#### `core/network/`
- **Chức năng**: Network layer với HTTP client, interceptors, API clients
- **Cấu trúc**:
  ```
  network/
  ├── api/              # API clients (affiliate, game, location, etc.)
  ├── interceptor/      # HTTP interceptors (auth, logger, error)
  ├── dio_http_service.dart
  ├── error_mapper.dart
  └── restful_api_client.dart
  ```
- **Lý do**: Tập trung tất cả network logic, dễ mock khi test

#### `core/storage/`
- **Chức năng**: Local storage và preferences
- **Files**:
  - `data_preference.dart` - SharedPreferences wrapper
  - `contact_storage_service.dart` - Contact storage
  - `point/` - Point management storage
- **Lý do**: Tách biệt storage logic, dễ thay đổi implementation

#### `core/services/`
- **Chức năng**: Core business services
- **Cấu trúc**:
  ```
  services/
  ├── auth/
  │   ├── login_service.dart
  │   ├── logout_service.dart
  │   └── token_refresh_service.dart
  └── notification/
  ```
- **Lý do**: Services được dùng chung, không thuộc về feature cụ thể

#### `core/platform/`
- **Chức năng**: Platform-specific code
- **Cấu trúc**:
  ```
  platform/
  ├── web/              # Web-specific (x-app-sdk, web helpers)
  ├── permission/       # Permissions (biometric, etc.)
  └── firebase/         # Firebase integration
  ```
- **Lý do**: Tách biệt platform code, dễ maintain cho từng platform

#### `core/base/`
- **Chức năng**: Base classes cho screens, viewmodels, models
- **Files**:
  - `base_screen.dart` - Base class cho screens
  - `base_view_model.dart` - Base class cho viewmodels
  - `base_response_model.dart` - Base class cho API responses
  - `basic_state.dart` - Base state management
- **Lý do**: DRY principle, giảm code duplication

#### `core/utils/`
- **Chức năng**: Utilities và extensions
- **Cấu trúc**:
  ```
  utils/
  ├── validation_utils.dart
  ├── direction_google_map.dart
  ├── router_gage.dart
  └── extensions/       # Extension methods
      ├── collection_extension.dart
      ├── color_extension.dart
      ├── date_format.dart
      └── ...
  ```
- **Lý do**: Gộp `shared/` và `utils/` lại, tránh confusion

#### `core/navigation/`
- **Chức năng**: Navigation và routing
- **Files**:
  - `app_navigator.dart` - Main navigator
  - `deferred_routes.dart` - Lazy loading routes
  - `directional_action_type.dart` - Direction actions
  - `directional_screen.dart` - Direction screens
- **Lý do**: Gộp `deferred/` và `directional/` lại, tất cả navigation ở một nơi

#### `core/theme/`
- **Chức năng**: Theme, colors, styles, images
- **Files**:
  - `colors.dart` (từ `base_color.dart`)
  - `text_styles.dart` (từ `text_style.dart`)
  - `button_styles.dart` (từ `button_style.dart`)
  - `images.dart` (từ `define_image.dart`)
- **Lý do**: Đổi tên từ `resources/` sang `theme/` cho rõ ràng hơn

#### `core/widgets/`
- **Chức năng**: Core reusable widgets
- **Files**:
  - `app_loading.dart`
  - `alert/` - Alert widgets
  - `custom_app_bar.dart`
  - `custom_empty_widget.dart`
  - Và các widgets dùng chung khác
- **Lý do**: Widgets được dùng chung trên toàn app

---

### 2. Features Layer (`lib/features/`)

**Mục đích**: Mỗi feature là một module độc lập, có thể phát triển và test riêng biệt.

#### Cấu trúc mỗi Feature

Mỗi feature có 3 layers:

```
feature_name/
├── data/                    # Data layer (outermost)
│   ├── models/             # Data models (API responses)
│   ├── repositories/       # Repository implementations
│   └── datasources/        # Remote/Local data sources
│
├── domain/                  # Domain layer (middle)
│   ├── entities/           # Business entities
│   └── usecases/           # Use cases (business logic)
│
└── presentation/            # Presentation layer (innermost)
    ├── screens/            # UI screens
    ├── viewmodels/        # ViewModels (state management)
    └── widgets/            # Feature-specific widgets
```

#### Ví dụ: Auth Feature

```
features/auth/
├── data/
│   ├── models/
│   │   ├── login_token_response_model.dart    # API response model
│   │   ├── profile_response_model.dart
│   │   └── user_agreement_model.dart
│   ├── repositories/
│   │   └── auth_repository_impl.dart          # Implements domain repository
│   └── datasources/
│       ├── auth_remote_datasource.dart        # API calls
│       └── auth_local_datasource.dart         # Local storage
│
├── domain/
│   ├── entities/
│   │   └── user.dart                          # Business entity (pure Dart)
│   ├── repositories/
│   │   └── auth_repository.dart                # Repository interface
│   └── usecases/
│       ├── login_usecase.dart                  # Business logic
│       ├── logout_usecase.dart
│       └── get_profile_usecase.dart
│
└── presentation/
    ├── screens/
    │   ├── login/
    │   │   ├── login_screen.dart
    │   │   └── login_viewmodel.dart
    │   ├── otp/
    │   └── create_pass/
    └── widgets/
        └── auth_form_widget.dart
```

**Lợi ích**:
- **Separation of concerns**: Mỗi layer có trách nhiệm riêng
- **Testability**: Có thể test từng layer độc lập
- **Dependency rule**: Inner layers không phụ thuộc outer layers
- **Reusability**: Domain layer có thể reuse cho web/mobile

#### Feature Organization

**1. Auth Feature** (`features/auth/`)
- Login, OTP, Create password
- Authentication logic

**2. Home Feature** (`features/home/`)
- Home screen (40 files)
- Dashboard, main content

**3. Voucher Feature** (`features/voucher/`)
- Voucher screens (46 files)
- Voucher management

**4. Transaction Feature** (`features/transaction/`)
- Transaction screens (25 files)
- Transaction history, details

**5. Campaign Feature** (`features/campaign/`)
- Campaign 7day
- Quiz campaign
- Register campaign
- Invite friend campaign

**6. Payment Feature** (`features/payment/`)
- Electric payment
- Mobile card
- Topup
- Traffic service

**7. Point Feature** (`features/point/`)
- History point
- History point cashback
- Daily checkin

**8. Profile Feature** (`features/profile/`)
- Personal
- Settings
- Change password
- Delete account
- Biometric

**9. Notification Feature** (`features/notification/`)
- Notification screens (12 files)

**10. Membership Feature** (`features/membership/`)
- Membership screens (10 files)

**11. Affiliate Feature** (`features/affiliate/`)
- Affiliate
- Affiliate brand detail

**12. Game Feature** (`features/game/`)
- Game
- VPlay game center

**13. Support Feature** (`features/support/`)
- Support
- FAQs
- News

**14. Common Feature** (`features/common/`)
- Splash
- Onboarding
- Webview
- QR code
- Popup manager
- Main tab screen

---

### 3. Shared Layer (`lib/shared/`)

**Mục đích**: Widgets và components được dùng chung giữa các features.

```
shared/
└── widgets/
    ├── custom_navigation_bar.dart
    ├── custom_point_text_tag.dart
    └── ...
```

**Lý do**: Khác với `core/widgets/`, `shared/widgets/` là các widgets được dùng bởi nhiều features nhưng không phải core functionality.

---

## Dependency Flow

```
Presentation Layer (UI)
    ↓ depends on
Domain Layer (Business Logic)
    ↓ depends on
Data Layer (API, Storage)
```

**Quy tắc**:
- Presentation chỉ phụ thuộc Domain
- Domain không phụ thuộc Presentation hay Data
- Data phụ thuộc Domain (implements interfaces)

---

## So sánh với cấu trúc cũ

### Cấu trúc cũ:
```
lib/
├── screen/          # 375+ files trong 1 thư mục
├── model/           # Chỉ có auth models
├── networking/      # Network layer
├── services/        # Services rời rạc
├── shared/          # Không rõ mục đích
├── utils/           # Trùng với shared
└── ...
```

**Vấn đề**:
- Khó tìm code
- Khó maintain
- Khó test
- Khó scale

### Cấu trúc mới:
```
lib/
├── core/            # Core functionality
│   ├── config/
│   ├── network/
│   ├── storage/
│   └── ...
├── features/        # Feature modules
│   ├── auth/
│   │   ├── data/
│   │   ├── domain/
│   │   └── presentation/
│   └── ...
└── shared/          # Shared widgets
```

**Lợi ích**:
- ✅ Dễ tìm code (theo feature)
- ✅ Dễ maintain (mỗi feature độc lập)
- ✅ Dễ test (test từng layer)
- ✅ Dễ scale (thêm feature mới dễ dàng)

---

## Migration Strategy

### Phase 1: Core Reorganization (Low Risk)
- Gộp configs, utils, navigation
- Không ảnh hưởng business logic
- Dễ rollback

### Phase 2: Network & Services (Medium Risk)
- Di chuyển networking và services
- Cần update imports
- Test kỹ API calls

### Phase 3: Feature-based (High Risk)
- Di chuyển screens theo features
- Tạo data/domain/presentation layers
- Cần test toàn bộ app

---

## Best Practices

1. **Mỗi feature độc lập**: Có thể develop và test riêng
2. **Dependency injection**: Dùng GetX hoặc provider
3. **Repository pattern**: Tách biệt data source và business logic
4. **Use cases**: Mỗi business action là một use case
5. **Entities**: Pure Dart classes, không phụ thuộc framework

---

## Ví dụ Migration

### Trước (Cấu trúc cũ):
```dart
// lib/screen/login/login_screen.dart
import 'package:mypoint_flutter_app/networking/restful_api_client.dart';
import 'package:mypoint_flutter_app/model/auth/login_token_response_model.dart';
import 'package:mypoint_flutter_app/services/login_service.dart';
```

### Sau (Cấu trúc mới):
```dart
// lib/features/auth/presentation/screens/login/login_screen.dart
import 'package:mypoint_flutter_app/features/auth/domain/usecases/login_usecase.dart';
import 'package:mypoint_flutter_app/features/auth/domain/entities/user.dart';
```

**Lợi ích**:
- Presentation không phụ thuộc trực tiếp vào API
- Dễ test (mock use case)
- Dễ thay đổi implementation

---

## Kết luận

Cấu trúc mới giúp:
- ✅ Code organization tốt hơn
- ✅ Dễ maintain và scale
- ✅ Dễ test
- ✅ Team collaboration tốt hơn
- ✅ Tuân thủ Clean Architecture principles

