# Migration Guide - Chuyển đổi sang Cấu trúc Mới

## Tổng quan

Guide này hướng dẫn từng bước để migrate từ cấu trúc cũ sang cấu trúc mới theo Clean Architecture.

## ⚠️ Lưu ý quan trọng

1. **Backup trước khi bắt đầu**: Script sẽ tự động tạo backup, nhưng nên commit code trước
2. **Chạy từng phase**: Không chạy tất cả phases cùng lúc
3. **Test sau mỗi phase**: Đảm bảo app vẫn hoạt động sau mỗi bước
4. **Update imports**: Sau mỗi phase, cần update imports

## Prerequisites

```bash
# Đảm bảo đã commit code
git status
git add .
git commit -m "Before migration to new structure"
```

## Phase 1: Core Reorganization (Low Risk)

### Mục tiêu
Tổ chức lại core layer: configs, utils, navigation, theme

### Các thay đổi
- `configs/` → `core/config/`
- `env_loader.dart` → `core/config/env_loader.dart`
- `core/app_initializer.dart` → `core/initialization/app_initializer.dart`
- `utils/` + `shared/` → `core/utils/`
- `extensions/` → `core/utils/extensions/`
- `deferred/` + `directional/` → `core/navigation/`
- `resources/` → `core/theme/` (và rename files)

### Các bước

#### Bước 1: Chạy migration script
```bash
./migrate_structure.sh phase1
```

#### Bước 2: Update imports
```bash
./update_imports.sh phase1
```

#### Bước 3: Update file references
Cần update các references trong code:

**Resources/Theme:**
```dart
// Trước
import 'package:mypoint_flutter_app/resources/base_color.dart';
final color = BaseColor.primary;

// Sau
import 'package:mypoint_flutter_app/core/theme/colors.dart';
final color = AppColors.primary;  // Cần rename class nếu cần
```

**Configs:**
```dart
// Trước
import 'package:mypoint_flutter_app/configs/api_paths.dart';

// Sau
import 'package:mypoint_flutter_app/core/config/api_paths.dart';
```

#### Bước 4: Test
```bash
flutter pub get
flutter analyze
flutter run
```

#### Bước 5: Fix lỗi
- Kiểm tra các lỗi import
- Update các references còn sót
- Test lại app

---

## Phase 2: Network & Services (Medium Risk)

### Mục tiêu
Tổ chức lại network layer, services, storage, và platform code

### Các thay đổi
- `networking/` → `core/network/`
- `services/` → `core/services/auth/`
- `preference/` → `core/storage/`
- `web/` → `core/platform/web/`
- `permission/` → `core/platform/permission/`
- `firebase/` → `core/platform/firebase/`

### Các bước

#### Bước 1: Chạy migration script
```bash
./migrate_structure.sh phase2
```

#### Bước 2: Update imports
```bash
./update_imports.sh phase2
```

#### Bước 3: Update API client references
```dart
// Trước
import 'package:mypoint_flutter_app/networking/api/affiliate_api.dart';

// Sau
import 'package:mypoint_flutter_app/core/network/api/affiliate_api.dart';
```

#### Bước 4: Update service references
```dart
// Trước
import 'package:mypoint_flutter_app/services/login_service.dart';

// Sau
import 'package:mypoint_flutter_app/core/services/auth/login_service.dart';
```

#### Bước 5: Update storage references
```dart
// Trước
import 'package:mypoint_flutter_app/preference/data_preference.dart';

// Sau
import 'package:mypoint_flutter_app/core/storage/data_preference.dart';
```

#### Bước 6: Update platform references
```dart
// Trước
import 'package:mypoint_flutter_app/web/web_helper.dart';

// Sau
import 'package:mypoint_flutter_app/core/platform/web/web_helper.dart';
```

#### Bước 7: Test
```bash
flutter pub get
flutter analyze
flutter run
# Test API calls, services, storage
```

---

## Phase 3: Feature-based Organization (High Risk)

### Mục tiêu
Tổ chức screens theo features với data/domain/presentation layers

### ⚠️ Cảnh báo
Phase này **KHÔNG tự động hoàn toàn**. Cần:
- Di chuyển screens thủ công
- Tạo data/domain layers
- Refactor code để tuân thủ Clean Architecture

### Các bước

#### Bước 1: Tạo feature structure
```bash
./migrate_structure.sh phase3
```

Script sẽ tạo cấu trúc thư mục cho tất cả features nhưng **KHÔNG di chuyển files**.

#### Bước 2: Di chuyển screens thủ công

**Ví dụ: Auth Feature**

```bash
# Di chuyển login screens
mkdir -p lib/features/auth/presentation/screens/login
mv lib/screen/login/* lib/features/auth/presentation/screens/login/

# Di chuyển OTP screens
mkdir -p lib/features/auth/presentation/screens/otp
mv lib/screen/otp/* lib/features/auth/presentation/screens/otp/

# Di chuyển create_pass screens
mkdir -p lib/features/auth/presentation/screens/create_pass
mv lib/screen/create_pass/* lib/features/auth/presentation/screens/create_pass/
```

**Ví dụ: Home Feature**

```bash
# Di chuyển home screens
mkdir -p lib/features/home/presentation/screens
mv lib/screen/home/* lib/features/home/presentation/screens/
```

#### Bước 3: Di chuyển models

```bash
# Di chuyển auth models
mv lib/model/auth/* lib/features/auth/data/models/
```

#### Bước 4: Update imports trong screens

Cần update imports trong các files đã di chuyển:

```dart
// Trước (trong login_screen.dart)
import 'package:mypoint_flutter_app/model/auth/login_token_response_model.dart';
import 'package:mypoint_flutter_app/services/login_service.dart';
import 'package:mypoint_flutter_app/networking/restful_api_client.dart';

// Sau
import 'package:mypoint_flutter_app/features/auth/data/models/login_token_response_model.dart';
import 'package:mypoint_flutter_app/core/services/auth/login_service.dart';
import 'package:mypoint_flutter_app/core/network/restful_api_client.dart';
```

#### Bước 5: Tạo Domain Layer (Optional nhưng khuyến nghị)

**Tạo entities:**
```dart
// lib/features/auth/domain/entities/user.dart
class User {
  final String id;
  final String name;
  final String email;
  
  User({required this.id, required this.name, required this.email});
}
```

**Tạo use cases:**
```dart
// lib/features/auth/domain/usecases/login_usecase.dart
class LoginUseCase {
  final AuthRepository repository;
  
  LoginUseCase(this.repository);
  
  Future<Either<Failure, User>> call(LoginParams params) async {
    return await repository.login(params.email, params.password);
  }
}
```

**Tạo repository interface:**
```dart
// lib/features/auth/domain/repositories/auth_repository.dart
abstract class AuthRepository {
  Future<Either<Failure, User>> login(String email, String password);
  Future<Either<Failure, void>> logout();
}
```

#### Bước 6: Tạo Data Layer

**Tạo repository implementation:**
```dart
// lib/features/auth/data/repositories/auth_repository_impl.dart
class AuthRepositoryImpl implements AuthRepository {
  final AuthRemoteDataSource remoteDataSource;
  final AuthLocalDataSource localDataSource;
  
  @override
  Future<Either<Failure, User>> login(String email, String password) async {
    try {
      final response = await remoteDataSource.login(email, password);
      return Right(response.toEntity());
    } catch (e) {
      return Left(ServerFailure(e.toString()));
    }
  }
}
```

#### Bước 7: Update routes

Cần update route definitions:

```dart
// Trước
GetPage(
  name: '/login',
  page: () => LoginScreen(),
)

// Sau
GetPage(
  name: '/login',
  page: () => const LoginScreen(),  // Import từ features/auth
)
```

#### Bước 8: Test toàn bộ app
```bash
flutter pub get
flutter analyze
flutter test
flutter run
# Test tất cả features
```

---

## Checklist Migration

### Phase 1 Checklist
- [ ] Backup created
- [ ] Migration script chạy thành công
- [ ] Imports updated
- [ ] `flutter pub get` chạy thành công
- [ ] `flutter analyze` không có lỗi
- [ ] App chạy được
- [ ] Test các chức năng cơ bản

### Phase 2 Checklist
- [ ] Migration script chạy thành công
- [ ] Imports updated
- [ ] API calls hoạt động
- [ ] Services hoạt động
- [ ] Storage hoạt động
- [ ] Platform code hoạt động
- [ ] `flutter analyze` không có lỗi
- [ ] App chạy được

### Phase 3 Checklist
- [ ] Feature structures created
- [ ] Screens đã di chuyển
- [ ] Models đã di chuyển
- [ ] Imports updated
- [ ] Routes updated
- [ ] Domain layer created (optional)
- [ ] Data layer created (optional)
- [ ] `flutter analyze` không có lỗi
- [ ] App chạy được
- [ ] Tất cả features test thành công

---

## Rollback Plan

Nếu có vấn đề, có thể rollback:

```bash
# Restore từ backup
cp -r .migration_backup_YYYYMMDD_HHMMSS/lib/* lib/

# Hoặc từ git
git reset --hard HEAD
```

---

## Common Issues & Solutions

### Issue 1: Import errors
**Solution**: Chạy `./update_imports.sh phaseX` và kiểm tra lại

### Issue 2: Missing files
**Solution**: Kiểm tra backup folder, restore files cần thiết

### Issue 3: Circular dependencies
**Solution**: Refactor code để tránh circular imports

### Issue 4: Build errors
**Solution**: 
```bash
flutter clean
flutter pub get
flutter pub upgrade
```

---

## Best Practices

1. **Commit sau mỗi phase**: Để dễ rollback nếu cần
2. **Test kỹ**: Test tất cả features sau mỗi phase
3. **Code review**: Review code sau mỗi phase
4. **Documentation**: Update documentation khi cần
5. **Team communication**: Thông báo team về changes

---

## Timeline ước tính

- **Phase 1**: 1-2 giờ (low risk)
- **Phase 2**: 2-4 giờ (medium risk)
- **Phase 3**: 1-2 tuần (high risk, cần refactor nhiều)

---

## Support

Nếu gặp vấn đề:
1. Kiểm tra backup folder
2. Review migration logs
3. Check import errors với `flutter analyze`
4. Rollback nếu cần

