import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:mypoint_flutter_app/extensions/string_extension.dart';
import 'package:mypoint_flutter_app/widgets/custom_app_bar.dart';
import '../../base/base_screen.dart';
import '../../base/basic_state.dart';
import '../../widgets/image_loader.dart';
import 'models/notification_detail_model.dart';
import 'notification_detail_viewmodel.dart';

class NotificationDetailScreen extends BaseScreen {
  const NotificationDetailScreen({super.key});

  @override
  State<NotificationDetailScreen> createState() => _NotificationDetailScreenState();
}

class _NotificationDetailScreenState extends BaseState<NotificationDetailScreen> with BasicState {
  final _viewModel = Get.put(NotificationDetailViewModel());

  @override
  void initState() {
    super.initState();
    NotificationDetailModel? notification;
    String? notificationId;
    String? title;
    String? body;

    final args = Get.arguments;
    if (args is Map) {
      notification = args['notification'];
      notificationId = args['notificationId'];
      title = args['title'];
      body = args['body'];
    }
    if (title.orEmpty.isNotEmpty && body.orEmpty.isNotEmpty) {
      notification ??= NotificationDetailModel(title: title, body: body);
    }
    if (notificationId == null && notification == null) {
      WidgetsBinding.instance.addPostFrameCallback((_) {
        Get.back();
      });
      return;
    }
    _viewModel.onShowAlertError = (message) {
      if (message.isEmpty) return;
      showAlertError(
        content: message,
        onConfirmed: () {
          Get.back();
        },
      );
    };
    _viewModel.fetchNotificationDetail(id: notificationId, data: notification);
  }

  @override
  Widget createBody() {
    return Scaffold(
      appBar: CustomAppBar.back(title: "Chi tiết thông báo"),
      body: Obx(
        () => Container(
          color: Colors.grey.shade100,
          padding: const EdgeInsets.all(16),
          child: IntrinsicHeight(
            child: Container(
              padding: const EdgeInsets.all(16),
              decoration: BoxDecoration(color: Colors.white, borderRadius: BorderRadius.circular(12)),
              child: Row(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  ClipRRect(
                    borderRadius: BorderRadius.circular(8),
                    child: loadNetworkImage(
                      url: _viewModel.notification.value?.workingSite?.avatar ?? "",
                      fit: BoxFit.cover,
                      width: 40,
                      height: 40,
                      placeholderAsset: 'assets/images/ic_logo.png',
                    ),
                  ),
                  const SizedBox(width: 12),
                  Expanded(
                    child: Column(
                      crossAxisAlignment: CrossAxisAlignment.start,
                      children: [
                        Text(
                          _viewModel.notification.value?.title ?? '',
                          style: const TextStyle(fontWeight: FontWeight.bold, fontSize: 16),
                        ),
                        const SizedBox(height: 6),
                        Text(_viewModel.notification.value?.body ?? '', style: const TextStyle(fontSize: 14)),
                        const SizedBox(height: 10),
                        if (_viewModel.notification.value?.createTime.orEmpty.isNotEmpty == true)
                          Text(
                            _timeAgo(_viewModel.notification.value?.createTime),
                            style: const TextStyle(color: Colors.grey, fontSize: 13),
                          ),
                      ],
                    ),
                  ),
                ],
              ),
            ),
          ),
        ),
      ),
    );
  }

  String _timeAgo(String? dateStr) {
    if (dateStr == null) return '';
    final time = DateTime.tryParse(dateStr);
    if (time == null) return '';
    final now = DateTime.now();
    final diff = now.difference(time);
    if (diff.inDays > 0) return '${diff.inDays} ngày trước';
    if (diff.inHours > 0) return '${diff.inHours} giờ trước';
    if (diff.inMinutes > 0) return '${diff.inMinutes} phút trước';
    return 'Vừa xong';
  }
}
