import 'package:get/get_rx/src/rx_types/rx_types.dart';
import 'package:mypoint_flutter_app/networking/restful_api_client_all_request.dart';
import '../../networking/restful_api_viewmodel.dart';
import 'model/affiliate_brand_model.dart';
import 'model/affiliate_category_model.dart';
import 'model/affiliate_category_type.dart';
import 'model/affiliate_product_top_sale_model.dart';
import 'model/cashback_overview_model.dart';

class AffiliateTabViewModel extends RestfulApiViewModel {
  final RxList<AffiliateBrandModel> affiliateBrands = <AffiliateBrandModel>[].obs;
  final RxList<AffiliateCategoryModel> affiliateCategories = <AffiliateCategoryModel>[].obs;
  final RxList<AffiliateCategoryModel> allAffiliateCategories = <AffiliateCategoryModel>[].obs;
  final RxList<AffiliateProductTopSaleModel> affiliateProducts = <AffiliateProductTopSaleModel>[].obs;
  final RxBool isLoading = false.obs;
  final Rxn<CashbackOverviewModel> overview = Rxn<CashbackOverviewModel>();
  void Function((List<AffiliateBrandModel>, String) data)? onShowAffiliateBrandPopup;

  bool get isDataAvailable {
    return affiliateBrands.isNotEmpty ||
        affiliateCategories.isNotEmpty ||
        affiliateProducts.isNotEmpty;
  }

  Future<void> refreshData({bool isShowLoading = true}) async {
    if (isShowLoading && isDataAvailable) return;
    if (isShowLoading) showLoading();
    await Future.wait([
      _getAffiliateOverview(),
      _getAffiliateBrandGetList(),
      _getAffiliateCategoryGetList(),
      _getAffiliateProductTopSale(),
    ]);
    if (isShowLoading) hideLoading();
  }

  Future<void> _getAffiliateBrandGetList() async {
    await callApi<List<AffiliateBrandModel>>(
      request: () => client.affiliateBrandGetList(),
      onSuccess: (data, _) {
        affiliateBrands.assignAll(data);
      },
      onFailure: (msg, _, __) async {
        affiliateBrands.clear();
      },
      withLoading: false,
    );
  }

  Future<void> _getAffiliateCategoryGetList() async {
    await callApi<List<AffiliateCategoryModel>>(
      request: () => client.affiliateCategoryGetList(),
      onSuccess: (data, _) {
        final category = AffiliateCategoryModel(
          code: AffiliateCategoryType.other,
          name: "Khác",
        );
        final results = data;
        allAffiliateCategories.assignAll(results);

        final limitedData = results.take(7).toList();
        limitedData.add(category);
        affiliateCategories.assignAll(limitedData);
      },
      onFailure: (msg, _, _) async {
        affiliateCategories.clear();
        allAffiliateCategories.clear();
      },
      withLoading: false,
    );
  }

  Future<void> _getAffiliateProductTopSale() async {
    await callApi<List<AffiliateProductTopSaleModel>>(
      request: () => client.affiliateProductTopSale(),
      onSuccess: (data, _) {
        affiliateProducts.assignAll(data);
      },
      onFailure: (msg, _, __) async {
        affiliateProducts.clear();
      },
      withLoading: false,
    );
  }

  Future<void> _getAffiliateOverview() async {
    await callApi<CashbackOverviewModel>(
      request: () => client.getCashBackOverview(),
      onSuccess: (data, _) {
        overview.value = data;
      },
      onFailure: (msg, _, __) async {
        overview.value = null;
      },
      withLoading: false,
    );
  }

  Future<void> affiliateBrandGetListBuyCategory(AffiliateCategoryModel category) async {
    await callApi<List<AffiliateBrandModel>>(
      request: () => client.affiliateBrandGetList(categoryCode: AffiliateCategoryModel.codeToJson(category.code)),
      onSuccess: (data, _) {
        if (data.isNotEmpty) {
          onShowAffiliateBrandPopup?.call((data, category.name ?? ''));
        }
      },
    );
  }
}