#!/bin/bash

# Script build Production (tối đa hoá nén/ẩn mã) để gửi cho T3 tích hợp host app
set -e

echo "🚀 Building Production package (max optimization)..."

# 1) Chuyển env sang PRODUCTION
./scripts/switch_env.sh prod

# 2) Clean & get deps (đảm bảo build fresh)
echo "🧹 Cleaning..."
flutter clean || true
rm -rf .dart_tool build || true
echo "📦 Getting packages..."
flutter pub get

# 3) Build web (ẩn mã JS tối đa, bỏ source maps, tắt service worker để tránh cache khi cần)
# - --release: bật dart2js minify + tree-shake
# - --no-source-maps: không phát hành sourcemaps (ẩn mã tối đa)
# - --pwa-strategy=none: tắt SW nếu không cần PWA để tránh cache khó debug khi embed
# - --dart-define=FLUTTER_WEB_USE_SKIA=false: ưu tiên HTML renderer để hạn chế sự cố CORS ảnh khi host
echo "🔨 Flutter build web (release, no source maps)..."
flutter build web \
  --release \
  --no-source-maps \
  --pwa-strategy=none \
  --dart-define=FLUTTER_WEB_USE_SKIA=false

echo "📥 Copying x-app-sdk bundle into build artifacts..."
./scripts/copy_x_app_sdk_bundle.sh

echo "✅ Build completed. Preparing artifact..."

# 4) Đóng gói artifact gọn để gửi cho T3
STAMP=$(date +%Y%m%d_%H%M%S)
OUT_DIR="web_prod_${STAMP}"
mkdir -p "$OUT_DIR"
cp -r build/web/* "$OUT_DIR/"

# Tuỳ chọn: loại bỏ file không cần thiết (nếu có)
find "$OUT_DIR" -name "*.map" -type f -delete || true

ZIP_FILE="${OUT_DIR}.zip"
echo "📦 Creating ${ZIP_FILE} ..."
zip -rq "$ZIP_FILE" "$OUT_DIR"

echo ""
echo "🎉 Production artifact is ready"
echo "   Folder: $OUT_DIR"
echo "   Zip:    $ZIP_FILE"
echo ""
echo "📌 Gợi ý gửi cho T3: chỉ gửi file .zip hoặc nội dung thư mục $OUT_DIR"
echo "   - Đã tối đa hoá ẩn mã (minified), không đính kèm source maps"
echo "   - Không bật service worker để tránh cache khi embed trong host app"
echo "   - Ưu tiên HTML renderer để giảm sự cố CORS ảnh"
