import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:intl/intl.dart';
import 'package:mypoint_flutter_app/core/utils/extensions/string_extension.dart';
import '../../shared/preferences/data_preference.dart';
import '../../core/theme/base_color.dart';
import '../../shared/router_gage.dart';
import '../../shared/widgets/image_loader.dart';
import 'models/membership_level_model.dart';

class MemberLevelHeaderWidget extends StatelessWidget {
  final MembershipLevelModel? level;

  const MemberLevelHeaderWidget({super.key, this.level});

  @override
  Widget build(BuildContext context) {
    final double screenWidth = MediaQuery.of(context).size.width;
    final double imageHeight = screenWidth / (1125 / 702);
    return Stack(
      clipBehavior: Clip.none,
      children: [
        loadNetworkImage(
          url: level?.images?.firstOrNull?.imageUrl ?? "",
          fit: BoxFit.cover,
          height: imageHeight,
          width: double.infinity,
          placeholderAsset: 'assets/images/bg_header_membership.png',
        ),
        Positioned(left: 16, right: 16, bottom: -36, child: _buildCardHeader()),
      ],
    );
  }

  Widget _buildCardHeader() {
    final name = DataPreference.instance.displayName;
    return Column(
      crossAxisAlignment: CrossAxisAlignment.center,
      children: [
        Row(
          children: [
            loadNetworkImage(
                url: "level?.logo",
                width: 116,
                height: 116,
                placeholderAsset: "assets/images/ic_logo_rank_member.png",
            ),
            const SizedBox(width: 6),
            Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(
                  name.toUpperCase(),
                  style: const TextStyle(fontSize: 22, fontWeight: FontWeight.bold, color: Colors.white),
                ),
                const SizedBox(height: 4),
                if ((level?.levelName ?? "").isNotEmpty)
                  Container(
                    padding: const EdgeInsets.symmetric(horizontal: 12, vertical: 4),
                    decoration: BoxDecoration(color: Colors.white, borderRadius: BorderRadius.circular(20)),
                    child: Text(
                      (level?.levelName ?? "").capitalizeWords(),
                      style: TextStyle(color: BaseColor.primary800, fontSize: 14, fontWeight: FontWeight.bold),
                    ),
                  ),
              ],
            ),
          ],
        ),
        const SizedBox(height: 6),
        // Progress bar
        _buildCardInfo(),
      ],
    );
  }

  Widget _buildCardInfo() {
    final int point = double.tryParse(level?.accumulatedCounter?.couterPointValue ?? "0")?.toInt() ?? 0;
    final int pointMax = double.tryParse(level?.upgradePointThreshold ?? "0")?.toInt() ?? 1;
    final int spending = double.tryParse(level?.accumulatedCounter?.couterGmvValue ?? "0")?.toInt() ?? 0;
    final int spendingMax = double.tryParse(level?.upgradeGmvThreshold ?? "0")?.toInt() ?? 1;
    return Container(
      padding: const EdgeInsets.symmetric(vertical: 12, horizontal: 16),
      decoration: BoxDecoration(
        color: Colors.white,
        borderRadius: BorderRadius.circular(16),
        boxShadow: [
          BoxShadow(color: Colors.black.withOpacity(0.3), blurRadius: 16, spreadRadius: 0, offset: const Offset(0, 6)),
        ],
      ),
      child: Row(
        children: [
          Expanded(
            child: GestureDetector(
              onTap: () {
                Get.toNamed(historyPointScreen);
              },
              behavior: HitTestBehavior.opaque,
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  const Text("Điểm xét hạng", style: TextStyle(fontSize: 13, color: Colors.black54)),
                  const SizedBox(height: 4),
                  Text(
                    "${formatNumber(point)}/${formatNumber(pointMax)}",
                    style: const TextStyle(fontWeight: FontWeight.bold, fontSize: 15),
                  ),
                  const SizedBox(height: 4),
                  ClipRRect(
                    borderRadius: BorderRadius.circular(4),
                    child: LinearProgressIndicator(
                      value: pointMax == 0 ? 0 : point / pointMax,
                      backgroundColor: Colors.grey.shade200,
                      color: Colors.orangeAccent,
                      minHeight: 6,
                    ),
                  ),
                ],
              ),
            ),
          ),
          const SizedBox(width: 16),
          Container(width: 1, height: 48, color: Colors.grey.shade300),
          const SizedBox(width: 16),
          // Chi tiêu
          Expanded(
            child: GestureDetector(
              onTap: () {
                Get.toNamed('/historyPointCashBackScreen');
              },
              behavior: HitTestBehavior.opaque,
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  const Text("Chi tiêu", style: TextStyle(fontSize: 13, color: Colors.black54)),
                  const SizedBox(height: 4),
                  Text(
                    "${formatNumber(spending)}/${formatNumber(spendingMax)}",
                    style: const TextStyle(fontWeight: FontWeight.bold, fontSize: 15),
                  ),
                  const SizedBox(height: 4),
                  ClipRRect(
                    borderRadius: BorderRadius.circular(4),
                    child: LinearProgressIndicator(
                      value: spendingMax == 0 ? 0 : spending / spendingMax,
                      backgroundColor: Colors.grey.shade200,
                      color: Colors.orangeAccent,
                      minHeight: 6,
                    ),
                  ),
                ],
              ),
            ),
          ),
        ],
      ),
    );
  }

  String formatNumber(int value) {
    return NumberFormat.decimalPattern('vi_VN').format(value);
  }
}
