import 'dart:typed_data';
import 'package:encrypt/encrypt.dart' as enc;

/// Giống Swift:
/// - `cipherHex`: chuỗi hex của dữ liệu đã mã hoá AES
/// - `secretKey`: chuỗi key (UTF-8), yêu cầu 16 bytes (AES-128)
class Crypto {
  final String cipherHex;
  final String secretKey;

  const Crypto({required this.cipherHex, required this.secretKey});

  /// Decrypt AES-128/ECB/PKCS7 từ hex -> String (UTF-8). Lỗi -> null.
  String? decryption() {
    try {
      final keyBytes = _normalizeKeyUtf8(secretKey, 16); // AES-128 = 16 bytes
      final dataBytes = _hexToBytes(cipherHex);

      final key = enc.Key(keyBytes);
      final aes = enc.AES(key, mode: enc.AESMode.ecb, padding: 'PKCS7');
      final encrypter = enc.Encrypter(aes);

      final decrypted = encrypter.decrypt(enc.Encrypted(dataBytes));
      // ignore: avoid_print
      print('Decrypted Text: $decrypted');
      return decrypted;
    } catch (e) {
      // ignore: avoid_print
      print('Decryption failed: $e');
      return null;
    }
  }

  /// Chuyển hex -> bytes
  Uint8List _hexToBytes(String hex) {
    final s = hex.replaceAll(RegExp(r'\s+'), '');
    if (s.length % 2 != 0) {
      throw const FormatException('Invalid hex length');
    }
    final result = Uint8List(s.length ~/ 2);
    for (var i = 0; i < s.length; i += 2) {
      result[i ~/ 2] = int.parse(s.substring(i, i + 2), radix: 16);
    }
    return result;
  }

  /// Key từ UTF-8, đảm bảo đúng `len` bytes: nếu thiếu thì pad 0x00, nếu dư thì cắt.
  Uint8List _normalizeKeyUtf8(String key, int len) {
    final raw = Uint8List.fromList(key.codeUnits); // UTF-8 code units (ASCII-safe)
    if (raw.length == len) return raw;
    if (raw.length > len) return Uint8List.fromList(raw.sublist(0, len));
    // pad 0x00 đến đủ độ dài
    final out = Uint8List(len);
    out.setRange(0, raw.length, raw);
    return out;
  }
}
