import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:get/get.dart';
import 'package:mypoint_flutter_app/screen/create_pass/signup_create_password_repository.dart';
import 'package:mypoint_flutter_app/widgets/back_button.dart';
import '../../resources/base_color.dart';
import 'create_pass_viewmodel.dart';

class CreatePasswordScreen extends StatelessWidget {
  final ICreatePasswordRepository repository;

  const CreatePasswordScreen({super.key, required this.repository});

  @override
  Widget build(BuildContext context) {
    final vm = Get.put(CreatePasswordViewModel(repository));
    final isNewPassObscure = true.obs;
    final isConfirmPassObscure = true.obs;

    return Scaffold(
      appBar: AppBar(
        centerTitle: true,
        leading: CustomBackButton(),
      ),
      body: SafeArea(
        child: Stack(
          children: [
            GestureDetector(
              onTap: () => FocusScope.of(context).unfocus(),
              child: SingleChildScrollView(
                padding: const EdgeInsets.all(16),
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.stretch,
                  children: [
                    const Text("Tạo mật khẩu", style: TextStyle(fontSize: 24, fontWeight: FontWeight.bold)),
                    const SizedBox(height: 32),
                    Obx(() {
                      return _buildPasswordField(
                        hintText: "Nhập mật khẩu mới",
                        obscureText: isNewPassObscure.value,
                        onToggleObscure: () => isNewPassObscure.value = !isNewPassObscure.value,
                        onChanged: vm.onNewPasswordChanged,
                      );
                    }),
                    const SizedBox(height: 16),
                    Obx(() {
                      return _buildPasswordField(
                        hintText: "Xác nhận lại mật khẩu",
                        obscureText: isConfirmPassObscure.value,
                        onToggleObscure: () => isConfirmPassObscure.value = !isConfirmPassObscure.value,
                        onChanged: vm.onConfirmPasswordChanged,
                      );
                    }),
                    const SizedBox(height: 8),
                    Obx(() {
                      final err = vm.errorMessage.value;
                      if (err.isEmpty) {
                        return const SizedBox.shrink();
                      } else {
                        return Text(err, style: const TextStyle(color: Colors.red));
                      }
                    }),
                    const SizedBox(height: 8),
                    _buildInfoGuide(icon: Icons.info_outline, text: "Mật khẩu gồm 6 chữ số"),
                    const SizedBox(height: 4),
                    _buildInfoGuide(icon: Icons.info_outline, text: "Không bao gồm dãy số trùng nhau"),
                    const SizedBox(height: 4),
                    _buildInfoGuide(icon: Icons.info_outline, text: "Không bao gồm dãy số liên tiếp"),
                  ],
                ),
              ),
            ),
            SizedBox.expand(),
            Positioned(left: 0, right: 0, bottom: 16, child: _buildContinueButton(vm)),
          ],
        ),
      ),
    );
  }

  Widget _buildContinueButton(CreatePasswordViewModel vm) {
    return Obx(() {
      final enabled = vm.isButtonEnabled.value;
      return Container(
        color: Colors.white,
        padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 8),
        child: ElevatedButton(
          onPressed: enabled ? vm.onSubmit : null,
          style: ElevatedButton.styleFrom(
            minimumSize: const Size.fromHeight(48),
            backgroundColor: enabled ? BaseColor.primary500 : BaseColor.second400,
            shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(8)),
          ),
          child: Text(
            "Tiếp tục",
            style: TextStyle(color: enabled ? Colors.white : Colors.white, fontWeight: FontWeight.bold, fontSize: 16),
          ),
        ),
      );
    });
  }

  Widget _buildInfoGuide({required IconData icon, required String text}) {
    return Row(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Icon(icon, color: BaseColor.second400, size: 24),
        SizedBox(width: 8),
        Expanded(child: Text(text, style: const TextStyle(color: BaseColor.second400, fontSize: 14))),
      ],
    );
  }

  Widget _buildPasswordField({
    required String hintText,
    required bool obscureText,
    required VoidCallback onToggleObscure,
    required ValueChanged<String> onChanged,
  }) {
    return TextField(
      inputFormatters: [LengthLimitingTextInputFormatter(6)],
      obscureText: obscureText,
      onChanged: onChanged,
      decoration: InputDecoration(
        prefixIcon: const Icon(Icons.password, color: BaseColor.second500),
        border: OutlineInputBorder(
          borderRadius: BorderRadius.circular(8),
          borderSide: const BorderSide(color: Colors.grey, width: 1),
        ),
        enabledBorder: OutlineInputBorder(
          borderRadius: BorderRadius.circular(8),
          borderSide: const BorderSide(color: Colors.grey, width: 1),
        ),
        focusedBorder: OutlineInputBorder(
          borderRadius: BorderRadius.circular(8),
          borderSide: const BorderSide(color: Colors.grey, width: 1),
        ),
        hintText: hintText,
        hintStyle: const TextStyle(color: BaseColor.second200),
        fillColor: Colors.white,
        filled: true,
        contentPadding: const EdgeInsets.symmetric(horizontal: 12, vertical: 14),
        suffixIcon: IconButton(
          icon: Icon(obscureText ? Icons.visibility_off : Icons.visibility, color: BaseColor.second500),
          onPressed: onToggleObscure,
        ),
      ),
    );
  }
}
