import 'package:get/get.dart';
import 'package:get/get_rx/src/rx_types/rx_types.dart';
import 'package:mypoint_flutter_app/networking/restful_api_client_all_request.dart';
import 'package:mypoint_flutter_app/preference/data_preference.dart';
import 'package:mypoint_flutter_app/screen/data_network_service/product_network_data_model.dart';
import 'package:mypoint_flutter_app/widgets/custom_toast_message.dart';
import '../../base/base_response_model.dart';
import '../../networking/restful_api_viewmodel.dart';
import '../../configs/constants.dart';
import '../../preference/contact_storage_service.dart';
import '../../preference/point/point_manager.dart';
import '../topup/models/brand_network_model.dart';
import '../voucher/models/product_brand_model.dart';

class DataNetworkServiceViewModel extends RestfulApiViewModel {
  final RxList<String> histories = <String>[].obs;
  final RxList<ProductBrandModel> topUpBrands = <ProductBrandModel>[].obs;
  final RxList<TopUpNetworkDataModel> topUpNetworkData = <TopUpNetworkDataModel>[].obs;
  final Map<String, List<TopUpNetworkDataModel>> _allValue = {};

  final Rxn<ProductBrandModel> selectedBrand = Rxn<ProductBrandModel>();
  final Rxn<ProductNetworkDataModel> selectedProduct = Rxn<ProductNetworkDataModel>();
  final RxString phoneNumber = ''.obs;
  void Function(String message)? onShowAlertError;
  void Function(String message)? onShowAlertRedeemSuccess;

  int get payPoint {
    return (int.tryParse(selectedProduct.value?.prices?.firstOrNull?.payPoint ?? "0") ?? 0);
  }
  bool get isValidBalance {
    return UserPointManager().point >= payPoint;
  }

  @override
  void onInit() {
    super.onInit();
    final myPhone = DataPreference.instance.phone ?? '';
    phoneNumber.value = myPhone;
    ContactStorageService().getUsedContacts().then((value) {
      if (value.isNotEmpty) {
        histories.assignAll(value);
      } else {
        histories.assignAll([myPhone]);
      }
    });
    if (!histories.contains(myPhone)) {
      histories.insert(0, myPhone);
      ContactStorageService().saveUsedContact(myPhone);
    }
  }

  Future<void> firstLoadNetworkData() async {
    await _getNetworkBrands();
  }

  Future<void> _getNetworkBrands() async {
    await callApi<List<ProductBrandModel>>(
      request: () => client.productTopUpBrands(),
      onSuccess: (data, _) {
        topUpBrands.assignAll(data);
        checkMobileNetwork();
      },
      onFailure: (mgs, _, _) async {
        topUpBrands.clear();
        onShowAlertError?.call(mgs);
      },
    );
  }

  Future<void> checkMobileNetwork() async {
    await callApi<BrandNameCheckResponse>(
      request: () => client.checkMobileNetwork(phoneNumber.value),
      onSuccess: (data, _) {
        final brandCode = data.brand ?? '';
        var brand = topUpBrands.isNotEmpty
            ? topUpBrands.firstWhere(
                (b) => b.code == brandCode,
                orElse: () => topUpBrands.first,
              )
            : topUpBrands.firstOrNull;
        selectedBrand.value = brand;
        getTelcoDetail();
      },
      onFailure: (_, _, _) async {
        final first = topUpBrands.firstOrNull;
        if (first != null) selectedBrand.value = first;
        getTelcoDetail();
      },
    );
  }

  Future<void> getTelcoDetail({String? selected}) async {
    final id = selectedBrand.value?.id;
    final code = selectedBrand.value?.code;
    if (id == null && code == null) return;

    void makeSelected(List<TopUpNetworkDataModel> data) {
      bool didSelect = false;
      final list = data.expand((e) => e.products ?? []).toList();
      if (selected != null && selected.isNotEmpty) {
        for (var item in list) {
          final isMatch = item.id == int.tryParse(selected);
          if (isMatch) {
            selectedProduct.value = item;
            didSelect = true;
          }
        }
      }
      if (!didSelect && selectedProduct.value == null) {
        selectedProduct.value = list.firstOrNull;
      }
    }
    // Dùng cache nếu có
    if (_allValue.containsKey(code)) {
      final cached = _allValue[code]!;
      topUpNetworkData.assignAll(cached);
      makeSelected(cached);
      return;
    }
    await callApi<List<TopUpNetworkDataModel>>(
      request: () => client.getNetworkProducts((id ?? 0).toString()),
      onSuccess: (data, _) {
        final filtered = (data)
            .where((e) => e.products?.isNotEmpty == true)
            .toList();
        _allValue[code ?? ""] = filtered;
        topUpNetworkData.assignAll(filtered);
        makeSelected(filtered);
      },
      onFailure: (_, _, _) async {},
    );
  }

  Future<void> redeemProductMobileCard() async {
    final id = selectedProduct.value?.id;
    if (id == null) {
      showToastMessage('Vui lòng chọn sản phẩm');
      return;
    }
    await callApi<EmptyCodable>(
      request: () => client.redeemProductTopUps(id, phoneNumber.value),
      onSuccess: (data, _) {
        onShowAlertRedeemSuccess?.call("Chúc mừng bạn đã đổi Ưu đãi data thành công");
      },
      onFailure: (msg, _, _) async {
        onShowAlertError?.call(msg.isNotEmpty ? msg : Constants.commonError);
      },
    );
  }
}